// ============================================================================
// JBGameRulesOvertimeLastChance
// Copyright 2005 by Michael "_Lynx" Sokolkov <unreal-pz@narod.ru>
// except FindRules function based on JBGameRulesLlamaHunt.FindLlamaHuntRules
// Date      : 03.29.2005
// Revision  : 12
//
// Prevents the player from activating the switches in overtime
// ============================================================================
class JBGameRulesOvertimeLastChance extends JBGameRules;

var bool bOvertimeStarted;								// flag var to check if there was an overtime
var bool bCountDownStarted;								// flag var to check if the match is starting
var bool bLastExecute;										// flag var to check if the execution is final
var bool bDebug;                          // Toggle debug messages

struct JailInfo{
 	var name thisJailTag;
 	var JBInfoJail thisJail;
};

var array<JailInfo> Jails;

var JBAddonOvertimeLastChance MyAddon;


// ============================================================================
// PostBeginPlay
//
// Just starts the timer
// ============================================================================

function PostBeginPlay(){
	Super.PostBeginPlay();
	SetTimer(0.5, True);
	if (myAddon != None)
  	bDebug = myAddon.bDebug;
}


// ============================================================================
// Timer
//
// Checks if the game went into overtime and if the overtime was before. If so,
// calls the OvertimeStarted function which manages all overtime stuff
// ============================================================================

function Timer()
{
  if (Deathmatch(Level.Game).bStartedCountDown && !bCountDownStarted){
  	RememberTags();
  	bCountDownStarted = true;
  	return;
  }

	if (!bOvertimeStarted && Level.Game.bOverTime){
    OvertimeStarted();
	}
}


// ============================================================================
// OvertimeStarted
//
// if the games goes into the overtime broadcasts the message informing all
// players that the locks are jammed. Then, spawns the blockers for each
// release switch and kills emitters client-side. Also, if set the players are
// restarted
// ============================================================================

function OvertimeStarted()
{
 	local JBGameObjectiveSwitch mySwitch;
 	local Controller C;
 	local JBTagPlayer firstTagPlayer;
 	local JBTagPlayer thisTagPlayer;

  // broadcasting "Locks jammed" message to all players
  BroadcastLocalizedMessage(class'JBLocalMessageOTLockJam', 100);


  if (myAddon.RestartPlayersInOT == 1){ RestartNonJailed();                 }
  if (myAddon.RestartPlayersInOT == 2){ Jailbreak(Level.Game).RestartAll(); }

	// spawn blockers
	Foreach AllActors(class'JBGameObjectiveSwitch', mySwitch){
		SpawnBlockerFor(mySwitch);
	}

  if (!myAddon.bAllowArenaInOT){

    for (C = Level.ControllerList; C.nextController != None; C = C.nextController){
      if (C.PlayerReplicationInfo != None){
        ExcludePlayerFromArena(C);
        thisTagPlayer = Class'JBTagPlayer'.Static.FindFor(C.PlayerReplicationInfo);

        if (thisTagPlayer.GetArenaPending() != None)
          thisTagPlayer.GetArena().MatchCancel();
      }
    }

  }

  firstTagPlayer = JBGameReplicationInfo(Level.Game.GameReplicationInfo).firstTagPlayer;

  for (thisTagPlayer = firstTagPlayer; thisTagPlayer != None; thisTagPlayer = thisTagPlayer.nextTag){
    if ( Bot(thisTagPlayer.GetController()) != None ){
      JBBotTeam(TeamGame(Level.Game).Teams[thisTagPlayer.GetTeam().TeamIndex].AI).PutOnFreelance(Bot(thisTagPlayer.GetController()));
    }
  }

  JBBotTeam( TeamGame(Level.Game).Teams[0].AI ).SetTactics('Aggressive');
  JBBotTeam( TeamGame(Level.Game).Teams[1].AI ).SetTactics('Aggressive');

	ClearTags();
  myAddon.bMakeChanges 	= True;
	bOvertimeStarted 			= True;
}


// ============================================================================
// CanRelease
//
// Called when a player attempts to release a team by activating a release
// switch. Returning False will prevent the release; in that case the
// objectives for this jail will remain disabled for a short time before
// they are activated again.
// Prevents a player from activating the switch in overtime. F.e. if the player
// literally sits on the switch when OT starts.
// ============================================================================

function bool CanRelease(TeamInfo Team, Pawn PawnInstigator, GameObjective Objective)
{
  local JBGameRules nextJBGameRules;

  nextJBGameRules = GetNextJBGameRules();

  if(bOvertimeStarted && PlayerController(PawnInstigator.Controller) != None){
  	Level.Game.BroadcastHandler.BroadcastLocalized(Self,PlayerController(PawnInstigator.Controller),class'JBLocalMessageOTLockJam', 200);  }

 	if (nextJBGameRules != None)
   	return nextJBGameRules.CanRelease(Team, PawnInstigator, Objective) && !bOvertimeStarted;

 	return false;
}


// ============================================================================
// NotifyExecutionCommit
//
// Called when a team is about to be executed, before the execution sequence
// starts and directly after the other players' views switch to the execution
// camera.
// Checks if the the given execution is final, and if so clears the changes.
// ============================================================================

function NotifyExecutionCommit(TeamInfo Team)
{
  local JBGameRules nextJBGameRules;

  if ((Level.Game.GameReplicationInfo.Teams[0].Score >= Level.Game.GoalScore ||
       Level.Game.GameReplicationInfo.Teams[1].Score >= Level.Game.GoalScore) && Level.Game.GoalScore > 0){
			myAddon.bUndoChanges = True;
			KillBlockers();
	}
  nextJBGameRules = GetNextJBGameRules();

  if (nextJBGameRules != None)
    nextJBGameRules.NotifyExecutionCommit(Team);
}


// ============================================================================
// CanSendToArena
//
// Called to check whether a jailed player can be sent to the given arena. If
// this function returns False during the arena countdown for a player already
// scheduled for a fight in the given arena, the match will be cancelled. The
// bForceSendToArena out parameter can be set to allow even players who are
// not in jail to be sent to the arena.
// ============================================================================

function bool CanSendToArena(JBTagPlayer TagPlayer, JBInfoArena Arena, out byte bForceSendToArena)
{
  local JBGameRules nextJBGameRules;
  local bool bForbid;

  if (Level.Game.bOverTime && !myAddon.bAllowArenaInOT)
    bForbid = true;

  nextJBGameRules = GetNextJBGameRules();
  if (nextJBGameRules != None)
    return nextJBGameRules.CanSendToArena(TagPlayer, Arena, bForceSendToArena) && !bForbid;

  return True && !bForbid;
}


//=============================================================================
// FindRules
// based on JBgameRulesLlamaHunt.FindLlamaHuntRules
//
// Returns an existing JBGameRulesOvertimeLastChance actor or spawns and
// registers a new one if nothing was found.
//=============================================================================

static function JBGameRulesOvertimeLastChance FindRules(Actor Requester)
{
  local JBGameRules thisJBGameRules;

  if ( Requester == None ) {
    // can't work without an actor reference
    Warn("No requesting actor specified.");
    return None;
  }

  if ( JailBreak(Requester.Level.Game) == None ) {
    // doesn't work without Jailbreak
    log("Not a Jailbreak game.", 'OvertimeLastChance');
    return None;
  }

  for (thisJBGameRules = JailBreak(Requester.Level.Game).GetFirstJBGameRules();
       thisJBGameRules != None;
       thisJBGameRules = thisJBGameRules.GetNextJBGameRules()) {
    if ( JBGameRulesOvertimeLastChance (thisJBGameRules) != None )
      return JBGameRulesOvertimeLastChance (thisJBGameRules);
  }

  // no JBGameRulesOvertimeLastChance found, spawn a new one and register it
  thisJBGameRules = Requester.Spawn(Default.Class);
  if ( Requester.Level.Game.GameRulesModifiers == None )
    Requester.Level.Game.GameRulesModifiers = thisJBGameRules;
  else
    Requester.Level.Game.GameRulesModifiers.AddGameRules(thisJBGameRules);

  return JBGameRulesOvertimeLastChance (thisJBGameRules);
}


// ============================================================================
// RestartNonJailed
//
// Iterates over all players and restart players that are not in Jail and Arena
// ============================================================================

function RestartNonJailed(){

  local JBTagPlayer firstTagPlayer;
  local JBTagPlayer thisTagPlayer;
  local JBTagPlayer nextTagPlayer;

  firstTagPlayer = JBGameReplicationInfo(Level.Game.GameReplicationInfo).firstTagPlayer;

  for (thisTagPlayer = firstTagPlayer; thisTagPlayer != None; thisTagPlayer = nextTagPlayer){
    nextTagPlayer = thisTagPlayer.nextTag;

    if (thisTagPlayer.GetJail() == None && thisTagPlayer.GetArena() == None){
      thisTagPlayer.RestartInFreedom();
    }
  }
}


// ============================================================================
// ExcludePlayerFromArena
//
// Iterates over all arenas and adds the given controller to exclusion list,
// so the given player can't participate in arena matches any more
// ============================================================================

function ExcludePlayerFromArena(controller theExcluded)
{
  local JBInfoArena firstArena;
  local JBInfoArena Arena;

  if ( myAddon.bAllowArenaInOT )
    return;

  firstArena = JBGameReplicationInfo(Level.Game.GameReplicationInfo).firstArena;
  for (Arena = firstArena; Arena != None; Arena = Arena.nextArena){
    Arena.ExcludeAdd(theExcluded);
  }
}


//=============================================================================
// RememberTags
//
// Utility function to remeber information on jailbreak info tags
// will be used in v2
//=============================================================================

function RememberTags(){

	local int i;
	local JBInfoJail myJailActor;

  foreach AllActors(class'JBInfoJail', myJailActor){
		i = Jails.Length;
		Jails.Length = i + 1;
		Jails[i].thisJail = myJailActor;
		Jails[i].thisJailTag = myJailActor.Tag;
  }
}


//=============================================================================
// ClearTags
//
// Utility function to clear JBInfoJail tags. That's a hack to make bots
// ignore the switch in overtime and go on offence.
//=============================================================================

function ClearTags(){

	local JBInfoJail myJailActor;

  foreach AllActors(class'JBInfoJail', myJailActor){
		myJailActor.Tag = '';
  }
}


//=============================================================================
// SpawnBlockerFor
//
// Utility function to spawn the blocker actor for switch.
//=============================================================================

function SpawnBlockerFor(JBGameObjectiveSwitch mySwitch){

 	local vector BlockerLocation;
 	local JBDecoOTSwitchBlocker myBlocker;

 	BlockerLocation 	= mySwitch.Location;
	BlockerLocation.Z = BlockerLocation.Z + 9.0;

	myBlocker = Spawn(class'JBDecoOTSwitchBlocker',self,,Blockerlocation,mySwitch.Rotation);
	myBlocker.RelatedSwitch = mySwitch;
}

//=============================================================================
// SpawnBlockerFor
//
// Utility function to spawn the blocker actor for switch.
//=============================================================================

function KillBlockers(){

	local JBDecoOTSwitchBlocker myBlocker;

	foreach AllActors(class'JBDecoOTSwitchBlocker', myBlocker)
		myBlocker.Destroy();

}

DefaultProperties
{
}
