class DC_Walker extends ONSHoverCraft placeable;

#exec OBJ LOAD FILE=..\Animations\DC_Walker_A.ukx
#exec OBJ LOAD FILE=..\Textures\DC_Walker_T.utx
#exec OBJ LOAD FILE=..\StaticMeshes\DC_Walker_C.usx
#exec OBJ LOAD FILE=..\Sounds\DC_Walker_S.uax
#exec OBJ LOAD FILE=..\Sounds\WeaponSounds.uax

var Walker_part Left_Slave, Right_Slave;
var WHinge      Left_Joint, Right_Joint;

var float   Parked_Upright, Walking_Upright, Falling_Upright;
var float   Rotating_Speed, ChasYaw_Buffer;
var bool    ChasAligned, ChasRoatating, ShowLock, KnockedOut;
var Rotator RelChasRot; // In-mesh chassis rotation, Yaw only

// Internal system of states instead enabling/disabling 'tick'...
var enum W_State {
                  W_Initialising, // at the begining when parts are spawning
                  W_Parked,  // legs stiffnes to minimum, but uprightstiffnes to max, no rotation...
                  W_Walking, // legs stiffnes to maximum, full steps and rotation procced...
                  W_Falling  // same as walking, but no legs controll - prepare to land...
                 } WalkerState;

var Walker_leg Left_Leg, Right_Leg; // actual S_parts wich are walking...
var vector     Mass_Center, Location_2D; // balance point between legs where to stand, if it changes - chassis moves...
var float      Balancing_force, Standing_force, OnGroundDelay;
var float      StandHeight, TempHeight, SitHeight, RiseSpeed, SpawnHeight;
var float      StepSize, CroachSize, KnockDowntime, RecoverTime;

// Network
// do step only if on server, clients must wait whait server comand
// comands are bool flags and new step locations

var vector NewLeftLocation, NewRightLocation;
var vector OldLeftLocation, OldRightLocation;

var TexRotator WalkDirRing;
var vector     Rocket_Offsets[20];
var name       Barrel[20];
var DC_Rocket  Rockets[20];
var byte       BarrelIndex, Rocket_Sensor[20], Barrel_Order[20];
var Vehicle    CurrentTarget;
var Emitter    XDamagedEffect;

replication
{
 reliable if (Role == ROLE_Authority)
  RelChasRot, NewLeftLocation, NewRightLocation,
  KnockedOut, Rocket_Sensor,   BarrelIndex,
  CurrentTarget;
}

simulated event PostBeginPlay()
{
  local rotator HingesRot, RocketsScroll;
  local vector  Left_SocketLoc, Right_SocketLoc;
  local byte    r;

  Left_SocketLoc = GetBoneCoords('left_leg_socket').Origin;
  Right_SocketLoc = GetBoneCoords('right_leg_socket').Origin;
  // make 90 deg pitch angle
  HingesRot.Yaw = Rotation.Yaw;
  HingesRot.Roll = Rotation.Roll;
  HingesRot.Pitch = Rotation.Pitch - 16384;

  // left slave
  Left_Slave = Spawn(class'Z_part', self,, Left_SocketLoc, Normalize(Rotation));
  Left_Slave.Master = self;
  Left_Slave.Walker_Craft = self;
  Left_Slave.PartType = PT_Left;
  Left_Slave.AmbientGlow = AmbientGlow;
  Left_Joint = Spawn(class'WHinge', none,, Left_SocketLoc, Normalize(HingesRot));
  Left_Joint.KConstraintActor1 = self;
  Left_Joint.KConstraintActor2 = Left_Slave;
  Left_Joint.Parked_Stiffness = 1800;
  Left_Joint.Walking_Stiffness = 1800;
  Left_Joint.Falling_Stiffness = 1800;
  Left_Joint.SetStiffness(2);
  // right slave
  Right_Slave = Spawn(class'Z_part', self,, Right_SocketLoc, Normalize(Rotation));
  Right_Slave.Master = self;
  Right_Slave.Walker_Craft = self;
  Right_Slave.PartType = PT_Right;
  Right_Slave.AmbientGlow = AmbientGlow;
  Right_Joint = Spawn(class'WHinge', none,, Right_SocketLoc, Normalize(HingesRot));
  Right_Joint.KConstraintActor1 = self;
  Right_Joint.KConstraintActor2 = Right_Slave;
  Right_Joint.Parked_Stiffness = 1800;
  Right_Joint.Walking_Stiffness = 1800;
  Right_Joint.Falling_Stiffness = 1800;
  Left_Joint.SetStiffness(2);
  // setup physics
  Left_Joint.SetPhysics(Phys_Karma);
  Right_Joint.SetPhysics(Phys_Karma);

  /// for HUD ///
  WalkDirRing = new class'TexRotator';
  WalkDirRing.Material = Texture'DC_Walker_T.HUD.Ring_plane';
  WalkDirRing.UOffset = 256;
  WalkDirRing.VOffset = 256;

  for (r=0; r<20; r++)
   {
    RocketsScroll.Roll = 16384 + r*32768;
    Rockets[r] = Spawn(class'DC_Rocket', self,, GetBoneCoords(Barrel[r]).Origin);
    SetBoneRotation(Barrel[r], RocketsScroll);
    AttachToBone(Rockets[r], Barrel[r]);
    Rockets[r].launched = false;
    Rockets[r].RocketIndex = r;
    Rocket_Sensor[r] = 2;
   }

  if (XDamagedEffect == none)
   {
    XDamagedEffect = Spawn(Class'DC_Walker.DamagedMultiEffect');
    AttachToBone(XDamagedEffect, 'chassis_bone');
   }

  Super.PostBeginPlay();
}

simulated event SetInitialState()
{
  Super.SetInitialState();

  WalkerState = W_Initialising;
  Enable('Tick');
}

simulated function Tick(float dt)
{
  if (WalkerState == W_Walking)
  {
   RotatingProcess(dt, false);
   HoldAboveGround(dt);
   InputProcessing(dt);
   BalanceProcess();
   if ((Rise != 0) && (bDriving != false) && (Bot(Controller) == none))
    {
     if (Rise < 0)
      {
       SetWState(W_Parked);
       PlaySound(Sound'DC_Walker_S.enddown_xeng',,255,,600);
      }
     else if ((Rise > 0) && (Left_Leg.IsOnGround() == true) && (Right_Leg.IsOnGround() == true))
      {
       KAddImpulse(vector(rotation+rot(12000,0,0))*200000, GetBoneCoords('left_leg_socket').Origin, 'left_leg_socket');
       KAddImpulse(vector(rotation+rot(12000,0,0))*200000, GetBoneCoords('right_leg_socket').Origin, 'right_leg_socket');
      }
    }
  }
 else if (WalkerState == W_Parked)
  {
   BalanceProcess(); // do balance here
   if ((ChasAligned != true) || (bDriving != false)) // align chassis when parked
    ChasAligned = RotatingProcess(dt, true);
   if ((Rise == 0) && (bDriving != false) && (RecoverTime < Level.TimeSeconds))
    {
     SetWState(W_Walking);
     PlaySound(Sound'DC_Walker_S.startup_xeng',,255,,600);
    }
  }
 else if (WalkerState == W_Falling)
  {
   if ((Left_Leg.OnGroundTime > OnGroundDelay) || (Right_Leg.OnGroundTime > OnGroundDelay))
    SetWState(W_Walking);
  }
 else if (WalkerState == W_Initialising)
  InitLegs();

 if (Level.NetMode == NM_Client)
  CheckForRockets();
 else if (bDriving != false)
  CheckForTarget();

 UpdateEffects();

 Super.Tick(dt);
}

simulated function UpdateEffects()
{
 local float dk;

 dk = Health/HealthMax;

 if (XDamagedEffect != none)
  {
   if (dk > 0.75)
    {
     XDamagedEffect.Emitters[0].Disabled = true;
     XDamagedEffect.Emitters[1].Disabled = true;
     XDamagedEffect.Emitters[2].Disabled = true;
     XDamagedEffect.Emitters[3].Disabled = true;
     XDamagedEffect.Emitters[4].Disabled = true;
     XDamagedEffect.Emitters[5].Disabled = true;
    }
   else if (dk > 0.65)
    {
     XDamagedEffect.Emitters[0].Disabled = true;
     XDamagedEffect.Emitters[1].Disabled = false;
     XDamagedEffect.Emitters[2].Disabled = true;
     XDamagedEffect.Emitters[3].Disabled = true;
     XDamagedEffect.Emitters[4].Disabled = true;
     XDamagedEffect.Emitters[5].Disabled = true;
    }
   else if (dk > 0.55)
    {
     XDamagedEffect.Emitters[0].Disabled = false;
     XDamagedEffect.Emitters[1].Disabled = false;
     XDamagedEffect.Emitters[2].Disabled = true;
     XDamagedEffect.Emitters[3].Disabled = true;
     XDamagedEffect.Emitters[4].Disabled = true;
     XDamagedEffect.Emitters[5].Disabled = true;
    }
   else if (dk > 0.45)
    {
     XDamagedEffect.Emitters[0].Disabled = false;
     XDamagedEffect.Emitters[1].Disabled = false;
     XDamagedEffect.Emitters[2].Disabled = false;
     XDamagedEffect.Emitters[3].Disabled = true;
     XDamagedEffect.Emitters[4].Disabled = true;
     XDamagedEffect.Emitters[5].Disabled = true;
    }
   else if (dk > 0.3)
    {
     XDamagedEffect.Emitters[0].Disabled = false;
     XDamagedEffect.Emitters[1].Disabled = false;
     XDamagedEffect.Emitters[2].Disabled = false;
     XDamagedEffect.Emitters[3].Disabled = false;
     XDamagedEffect.Emitters[4].Disabled = true;
     XDamagedEffect.Emitters[5].Disabled = true;
    }
   else if (dk > 0.2)
    {
     XDamagedEffect.Emitters[0].Disabled = false;
     XDamagedEffect.Emitters[1].Disabled = false;
     XDamagedEffect.Emitters[2].Disabled = false;
     XDamagedEffect.Emitters[3].Disabled = false;
     XDamagedEffect.Emitters[4].Disabled = false;
     XDamagedEffect.Emitters[5].Disabled = true;
    }
   else
    {
     XDamagedEffect.Emitters[0].Disabled = false;
     XDamagedEffect.Emitters[1].Disabled = false;
     XDamagedEffect.Emitters[2].Disabled = false;
     XDamagedEffect.Emitters[3].Disabled = false;
     XDamagedEffect.Emitters[4].Disabled = false;
     XDamagedEffect.Emitters[5].Disabled = false;
    }
  }
}

// Do state changing effects here, still all local, synch will be in other place
simulated function SetWState(W_State NewState)
{
 if (NewState == W_Parked)         // PARKING START
  {
   WalkerState = W_Parked;
   SetStiffness(0); // parked stiffness
   KSetStayUprightParams(Parked_Upright, Default.UprightDamping);
   Balancing_force = Default.Balancing_force;
   ChasAligned = false;
   Left_Leg.SetLState(L_Falling);
   Right_Leg.SetLState(L_Falling);
   TempHeight = SitHeight;  // reset for next rise
  }
 else  if (NewState == W_Walking)  // WALKING START
  {
   WalkerState = W_Walking;
   KnockedOut = false;
   SetStiffness(1);  // walking stiffness
   KSetStayUprightParams(Walking_Upright, Default.UprightDamping);
   Left_Leg.SetLState(L_Falling);
   Right_Leg.SetLState(L_Falling);
  }
 else  if (NewState == W_Falling)  // FALLING START
  {
   WalkerState = W_Falling;
   SetStiffness(2);
   KSetStayUprightParams(Falling_Upright, Default.UprightDamping);
   Left_Leg.SetLState(L_Falling);
   Right_Leg.SetLState(L_Falling);
  }
}

////////// Walk solving structure here ////////////
simulated function InputProcessing(float dt)
{
 if (Throttle == 0)
  {
   if (Steering == 0)
    Positioning();  // this for no input, only positioning/stoping here
   else
    TurnOnPlace(Steering);
  }
 else if ((Throttle > 0) || (Steering == 0))
  TurnWalk(Throttle);
 else
  TurnOnPlace(-Steering);
 if (Level.NetMode == NM_Client)
  NetWorkStep();
}

simulated function NetWorkStep()
{
 if (OldLeftLocation != NewLeftLocation)
  {
   OldLeftLocation = NewLeftLocation;
   Left_Leg.StepTo(NewLeftLocation, VSize(NewLeftLocation - Left_Leg.Location)/4);
  }
 if (OldRightLocation != NewRightLocation)
  {
   OldRightLocation = NewRightLocation;
   Right_Leg.StepTo(NewRightLocation, VSize(NewRightLocation - Right_Leg.Location)/4);
  }
}

simulated function TurnWalk(float WalkDirection)
{
 local vector NewPosition;
 local vector  BestLeftLocation, BestRightLocation, LeftNormal, RightNormal;
 local vector  LeftStartTrace, RightStartTrace, LeftEndTrace, RightEndTrace;
 local Actor   LeftGround, RightGround;
 local float   RightOffset, LeftOffSet;
 local rotator LeftStartRot, RightStartRot;

 if (WalkDirection > 0)
  {
   NewPosition = vector(Rotation)*StepSize*1.5;
   Left_Leg.NewStepSpeed = 800;
   Right_Leg.NewStepSpeed = 800;
  }
 else if (WalkDirection < 0)
  {
   NewPosition = -vector(Rotation)*StepSize;
   Left_Leg.NewStepSpeed = 400;
   Right_Leg.NewStepSpeed = 400;
  }

 Left_Joint.SetStiffness(1);
 Right_Joint.SetStiffness(1);

 LeftStartRot.Yaw = Left_Leg.Rotation.Yaw + 16384;
 RightStartRot.Yaw = Right_Leg.Rotation.Yaw - 16384;

 if (Steering > 0)
  {
   LeftStartTrace = Right_Leg.Location + 1.5*vector(RightStartRot)*CroachSize + NewPosition/1.5;
   LeftStartTrace.Z = Location.Z;
   LeftEndTrace.X = LeftStartTrace.X;
   LeftEndTrace.Y = LeftStartTrace.Y;
   LeftEndTrace.Z = LeftStartTrace.Z - StandHeight*2;

   RightStartTrace = Left_Leg.Location + NewPosition;
   RightStartTrace.Z = Location.Z;
   RightEndTrace.X = RightStartTrace.X;
   RightEndTrace.Y = RightStartTrace.Y;
   RightEndTrace.Z = RightStartTrace.Z - StandHeight*2;
  }
 else if (Steering < 0)
  {
   LeftStartTrace = Right_Leg.Location + NewPosition;
   LeftStartTrace.Z = Location.Z;
   LeftEndTrace.X = LeftStartTrace.X;
   LeftEndTrace.Y = LeftStartTrace.Y;
   LeftEndTrace.Z = LeftStartTrace.Z - StandHeight*2;

   RightStartTrace = Left_Leg.Location + 1.5*vector(LeftStartRot)*CroachSize + NewPosition/1.5;
   RightStartTrace.Z = Location.Z;
   RightEndTrace.X = RightStartTrace.X;
   RightEndTrace.Y = RightStartTrace.Y;
   RightEndTrace.Z = RightStartTrace.Z - StandHeight*2;
  }
 else if (Steering == 0)
  {
   LeftStartTrace = Right_Leg.Location + vector(RightStartRot)*CroachSize + NewPosition;
   LeftStartTrace.Z = Location.Z;
   LeftEndTrace.X = LeftStartTrace.X;
   LeftEndTrace.Y = LeftStartTrace.Y;
   LeftEndTrace.Z = LeftStartTrace.Z - StandHeight*2;

   RightStartTrace = Left_Leg.Location + vector(LeftStartRot)*CroachSize + NewPosition;
   RightStartTrace.Z = Location.Z;
   RightEndTrace.X = RightStartTrace.X;
   RightEndTrace.Y = RightStartTrace.Y;
   RightEndTrace.Z = RightStartTrace.Z - StandHeight*2;
  }

 LeftGround=Trace(BestLeftLocation, LeftNormal, LeftEndTrace, LeftStartTrace);
 RightGround=Trace(BestRightLocation, RightNormal, RightEndTrace, RightStartTrace);

 if ((LeftGround != none) && (RightGround != none))
  {
   LeftOffSet = VSize(BestLeftLocation - Left_Leg.Location);
   RightOffset = VSize(BestRightLocation - Right_Leg.Location);

   if ( (LeftOffSet > 64) || (RightOffset > 64) )
    {
       if ((Left_Leg.OnGroundTime > Right_Leg.OnGroundTime) && (Right_Leg.OnGroundTime > OnGroundDelay))
        {
         if (Level.NetMode != NM_Client)
          {
           Left_Leg.StepTo(BestLeftLocation, LeftOffSet/4);
           NewLeftLocation = BestLeftLocation;
          }
        }
       else if ((Right_Leg.OnGroundTime >= Left_Leg.OnGroundTime) && (Left_Leg.OnGroundTime > OnGroundDelay))
        {
         if (Level.NetMode != NM_Client)
          {
           Right_Leg.StepTo(BestRightLocation, RightOffSet/4);
           NewRightLocation = BestRightLocation;
          }
        }
    }
  }
}

simulated function TurnOnPlace(float TurnSide)
{
 local vector  BestLeftLocation, BestRightLocation, LeftNormal, RightNormal;
 local rotator LeftStartRot, RightStartRot;
 local vector  LeftStartTrace, RightStartTrace, LeftEndTrace, RightEndTrace;
 local Actor   LeftGround, RightGround;
 local float   RightOffset, LeftOffSet;

 Left_Leg.NewStepSpeed = 250;
 Right_Leg.NewStepSpeed = 250;

 Left_Joint.SetStiffness(2);
 Right_Joint.SetStiffness(2);

 LeftStartRot = rotator(Left_Leg.Location - Right_Leg.Location);
 RightStartRot = rotator(Right_Leg.Location - Left_Leg.Location);
 if (TurnSide > 0)
  {
   LeftStartRot.Yaw = LeftStartRot.Yaw - 8192;
   RightStartRot.Yaw = RightStartRot.Yaw - 8192;
  }
 else
  {
   LeftStartRot.Yaw = LeftStartRot.Yaw + 8192;
   RightStartRot.Yaw = RightStartRot.Yaw + 8192;
  }

 LeftStartTrace = Right_Leg.Location + vector(LeftStartRot)*CroachSize*1.3;
 LeftStartTrace.Z = LeftStartTrace.Z + StandHeight;
 LeftEndTrace.X = LeftStartTrace.X;
 LeftEndTrace.Y = LeftStartTrace.Y;
 LeftEndTrace.Z = LeftStartTrace.Z - StandHeight*2;

 RightStartTrace = Left_Leg.Location + vector(RightStartRot)*CroachSize*1.3;
 RightStartTrace.Z = RightStartTrace.Z + StandHeight;
 RightEndTrace.X = RightStartTrace.X;
 RightEndTrace.Y = RightStartTrace.Y;
 RightEndTrace.Z = RightStartTrace.Z - StandHeight*2;

 LeftGround=Trace(BestLeftLocation, LeftNormal, LeftEndTrace, LeftStartTrace);
 RightGround=Trace(BestRightLocation, RightNormal, RightEndTrace, RightStartTrace);

 if ((LeftGround != none) && (RightGround != none))
  {
   LeftOffSet = VSize(BestLeftLocation - Left_Leg.Location);
   RightOffset = VSize(BestRightLocation - Right_Leg.Location);

   if ( (LeftOffSet > 64) || (RightOffset > 64) )
    {
     if ((Left_Leg.OnGroundTime > OnGroundDelay) && (Right_Leg.OnGroundTime > OnGroundDelay))
      {
       if ( Left_Leg.OnGroundTime >  Right_Leg.OnGroundTime )
        {
         if (Level.NetMode != NM_Client)
          {
           Left_Leg.StepTo(BestLeftLocation, LeftOffSet/3);
           NewLeftLocation = BestLeftLocation;
          }
        }
       else
        {
         if (Level.NetMode != NM_Client)
          {
           Right_Leg.StepTo(BestRightLocation, RightOffSet/3);
           NewRightLocation = BestRightLocation;
          }
        }
      }
    }
  }
}

simulated function Positioning()
{
 local vector  BestLeftLocation, BestRightLocation, LeftNormal, RightNormal;
 local rotator LeftStartRot, RightStartRot;
 local vector  LeftStartTrace, RightStartTrace, LeftEndTrace, RightEndTrace;
 local Actor   LeftGround, RightGround;
 local float   RightOffset, LeftOffSet;

 Left_Leg.NewStepSpeed = 250;
 Right_Leg.NewStepSpeed = 250;

 Left_Joint.SetStiffness(2);
 Right_Joint.SetStiffness(2);

 LeftStartRot.Pitch = Rotation.Pitch;
 LeftStartRot.Roll = Rotation.Roll;
 LeftStartRot.Yaw = Rotation.Yaw - 16384;
 LeftStartTrace = Location + vector(LeftStartRot)*CroachSize/2;
 LeftEndTrace.X = LeftStartTrace.X;
 LeftEndTrace.Y = LeftStartTrace.Y;
 LeftEndTrace.Z = LeftStartTrace.Z - StandHeight*2;

 RightStartRot.Pitch = Rotation.Pitch;
 RightStartRot.Roll = Rotation.Roll;
 RightStartRot.Yaw = Rotation.Yaw + 16384;
 RightStartTrace = Location + vector(RightStartRot)*CroachSize/2;
 RightEndTrace.X = RightStartTrace.X;
 RightEndTrace.Y = RightStartTrace.Y;
 RightEndTrace.Z = RightStartTrace.Z - StandHeight*2;

 LeftGround=Trace(BestLeftLocation, LeftNormal, LeftEndTrace, LeftStartTrace);
 RightGround=Trace(BestRightLocation, RightNormal, RightEndTrace, RightStartTrace);

 if ((LeftGround != none) && (RightGround != none))
  {
   LeftOffSet = VSize(BestLeftLocation - Left_Leg.Location);
   RightOffset = VSize(BestRightLocation - Right_Leg.Location);

   if ( (LeftOffSet > 64) || (RightOffset > 64) )
    {
     if ( (Right_Leg.OnGroundTime > OnGroundDelay) && (Left_Leg.OnGroundTime > OnGroundDelay) )
      {
       if (Left_Leg.OnGroundTime > Right_Leg.OnGroundTime)
        {
         if (Level.NetMode != NM_Client)
          {
           Left_Leg.StepTo(BestLeftLocation, LeftOffSet/3);
           NewLeftLocation = BestLeftLocation;
          }
        }
       else
        {
         if (Level.NetMode != NM_Client)
          {
           Right_Leg.StepTo(BestRightLocation, RightOffSet/3);
           NewRightLocation = BestRightLocation;
          }
        }
      }
    }
  }
}

////////////////////////////////////////////////////////

simulated event DrivingStatusChanged()
{
 if (bDriving == true)
  {
   SetWState(W_Walking);
   Left_Slave.SoundVolume = 128;
   Right_Slave.SoundVolume = 128;
  }
 else
  {
   SetWState(W_Parked);
   Left_Slave.SoundVolume = 0;
   Right_Slave.SoundVolume = 0;
  }
}

simulated function Destroyed()
{
 local byte r;

 Spawn(Class'DC_Walker.DC_Explosion',,,Location, Rotation);
 PlaySound(sound'WeaponSounds.BExplosion3',,255);

 Left_Joint.Destroy();
 Right_Joint.Destroy();

 Left_Slave.Destroy();
 Right_Slave.Destroy();

 for (r=0; r<20; r++)
  {
   if (Rockets[r] != none)
    Rockets[r].Destroy();
  }

 if (XDamagedEffect != none)
  XDamagedEffect.Destroy();

 Super.Destroyed();
}

simulated function SetStiffness(byte WHS_Mode)
{
 Z_part(Left_Slave).SetStiffness(WHS_Mode);
 Z_part(Right_Slave).SetStiffness(WHS_Mode);
}

// Rotation of vehicle it self will not change, so we take new
// headrotation to achieve from difference between view rotation
// and vehicle rotation. Vehicle rotation is controlled by legs only
simulated function bool RotatingProcess(float dt, optional bool Align)
{
 local float NewChasYaw, ChasYaw, DeltaYaw;
 local Actor AITarget;

 ChasYaw = Normalize(RelChasRot).Yaw;
 if ((Align != true) || (bDriving != false))
  {
   if (PlayerController(Controller) != none)
    {
     NewChasYaw = Normalize(Rotation).Yaw - Normalize(Controller.Rotation).Yaw;
     if ((abs(NewChasYaw - ChasYaw) < 3000) || (abs(NewChasYaw - ChasYaw) > 62536))
      ShowLock = true;
     else
      ShowLock = false;
    }
   else if (Bot(Controller) != none)
    {
     AITarget = none;
     if (Bot(Controller).Enemy != none)
      AITarget = Bot(Controller).Enemy;
     else if (Bot(Controller).Focus != none)
      AITarget = Bot(Controller).Focus;
     if (AITarget != none)
      NewChasYaw = Normalize(Rotation).Yaw - rotator(AITarget.Location - Location).Yaw;
     else
      NewChasYaw = Normalize(Rotation).Yaw -  Normalize(Controller.GetViewRotation()).Yaw;
    }
  }
 else
  {
   NewChasYaw = 0; // align when parking
   if ( abs(NewChasYaw - ChasYaw) < 0.1 )
     return true;
  }
 if (abs(NewChasYaw - ChasYaw) < 32768)
  DeltaYaw = (NewChasYaw - ChasYaw);
 else
  {
   if (NewChasYaw > 0)
    DeltaYaw = (NewChasYaw - ChasYaw - 65536);
   else
    DeltaYaw = (NewChasYaw - ChasYaw + 65536);
  }

  if (DeltaYaw > 0)
   RelChasRot.Yaw = ChasYaw + Min(Rotating_Speed, abs(DeltaYaw)*5)*dt;
  else if (DeltaYaw < 0)
   RelChasRot.Yaw = ChasYaw - Min(Rotating_Speed, abs(DeltaYaw)*5)*dt;

 if (abs(DeltaYaw) < Rotating_Speed*dt)
  SoundVolume = 0;
 else
  SoundVolume = 128;

 SetBoneRotation('chassis_bone', RelChasRot);
 return false;
}

// a little impulse to base to keep it centered
simulated function BalanceProcess()
{
 local vector Balance_impulse;

 Mass_Center = (Left_Leg.Location + Right_Leg.Location)/2;
 Location_2D = GetBoneCoords('base_bone').Origin;
 Mass_Center.Z = 0; // balance in 2D only
 Location_2D.Z = 0;
 Balance_impulse = (Mass_Center - Location_2D)*Balancing_force;
 KAddImpulse(Balance_impulse, GetBoneCoords('base_bone').Origin, 'base_bone');
}

// a little impulse to base to keep higher, returns false if falling
simulated function bool HoldAboveGround(float dt)
{
 local vector Stand_impulse, GroundTraced;
 local vector StartTrace, EndTrace;
 local float  AboveGround;
 local Actor  Ground;

 StartTrace = GetBoneCoords('base_bone').Origin;
 EndTrace = Left_Leg.Location + Right_Leg.Location - StartTrace;
 Ground = Trace(GroundTraced, Stand_impulse, EndTrace, StartTrace);
 if (Ground == none)
  return false;
 if (abs(TempHeight - StandHeight) > 1)
  {
   if (TempHeight < StandHeight)
    TempHeight = TempHeight + dt*RiseSpeed;
   else
    TempHeight = TempHeight - dt*RiseSpeed;
  }
 AboveGround = abs(GroundTraced.Z - StartTrace.Z);
 if ( abs(AboveGround - TempHeight) > 1 )
  {
   Stand_impulse.Z = (GroundTraced.Z + TempHeight - StartTrace.Z)*Standing_force;
   if (Left_Leg.LegState != L_Falling)
    KAddImpulse(Stand_impulse, GetBoneCoords('base_bone').Origin, 'right_leg_socket');
   if (Right_Leg.LegState != L_Falling )
    KAddImpulse(Stand_impulse, GetBoneCoords('base_bone').Origin, 'right_leg_socket');
  }
 return true;
}

// Legs initialisaton here, run in the begining untill legs not spawned
simulated function InitLegs()
{
 if ((Left_Slave != none) && (Right_Slave != none) )   // Z_parts
   {
   if ( (Z_part(Left_Slave).Slave != none) && (Z_part(Right_Slave).Slave != none) ) // B_parts
    {
      if ( (B_part(Z_part(Left_Slave).Slave).Slave != none) && (B_part(Z_part(Right_Slave).Slave).Slave != none) ) // H_parts
       {
         if ( (H_part(B_part(Z_part(Left_Slave).Slave).Slave).Slave != none) && (H_part(B_part(Z_part(Right_Slave).Slave).Slave).Slave != none) ) // S_parts
          {
           Left_Leg = Walker_leg(H_part(B_part(Z_part(Left_Slave).Slave).Slave).Slave);
           Left_Leg.Walker_Craft = self;
           Right_Leg = Walker_leg(H_part(B_part(Z_part(Right_Slave).Slave).Slave).Slave);
           Right_Leg.Walker_Craft = self;

           SetWState(W_Parked);
          }
       }
    }
  }
}

// put driver always on top of vehicle
function bool PlaceExitingDriver()
{
 local vector ExitLocation, RandomHole;

 RandomHole.X = 128 - Rand(255); RandomHole.Y = 128 - Rand(255);
 ExitLocation = (GetBoneCoords('rocket_base_2').Origin + GetBoneCoords('rocket_base_12').Origin)/2 + 1.5*Driver.default.CollisionRadius*Normal(RandomHole >> (Rotation - RelChasRot));

 Driver.SetLocation(ExitLocation);
 return true;
}

simulated event TeamChanged()
{
 Super.TeamChanged();
 Left_Slave.TeamChanged(Team);
 Right_Slave.TeamChanged(Team);
}

function TakeDamage(int Damage, Pawn instigatedBy, Vector Hitlocation, Vector Momentum, class<DamageType> DamageType)
{
 if (VSize(Momentum) > 80000)
  {
   RecoverTime = Level.TimeSeconds + KnockDowntime;
   SetWState(W_Parked);
   KnockedOut = true;
   PlaySound(Sound'DC_Walker_S.restart_xeng',,255,,600);
  }

 Super.TakeDamage(Damage, instigatedBy, Hitlocation, Momentum, damageType);
}

simulated event PostNetReceive()
{
 SetBoneRotation('chassis_bone', RelChasRot);

 Super.PostNetReceive();
}

simulated function DrawHUD(Canvas Canvas)
{
 local int    CentXStart, CentYStart, i, Test_Health, RocketsCount;
 local vector ScreenPos, BoxSize, TestOffSet;
 local float  HudDistK;

 Canvas.DrawColor.B = 0;
 Canvas.DrawColor.R = 0;
 Canvas.DrawColor.G = 255;
 Canvas.DrawColor.A = 128;
 // draw walk direction ring
 CentXStart = Canvas.SizeX/2 - 224; // cut 32 couse shit is visible
 CentYStart = Canvas.SizeY/2 - 224;
 WalkDirRing.Rotation.Yaw = -RelChasRot.Yaw;
 Canvas.SetPos(CentXStart, CentYStart);
 Canvas.DrawTile(WalkDirRing, 448, 448, 32, 32, 448, 448);
 // drawing croshairs
 CentXStart = Canvas.SizeX/2 - 256;
 CentYStart = Canvas.SizeY/2 - 64;
 Canvas.SetPos(CentXStart, CentYStart);
 if (ShowLock == true)
  {
   Canvas.DrawColor.R = 0;
   Canvas.DrawColor.G = 255;
  }
 else
  {
   Canvas.DrawColor.R = 255;
   Canvas.DrawColor.G = 0;
  }
 Canvas.DrawTile(Texture'DC_Walker_T.HUD.Cross_plane', 512, 128, 0.0, 0.0, 512, 128);
 // draw robot
 CentXStart = Canvas.SizeX - Canvas.SizeY/3;
 CentYStart = Canvas.SizeY - Canvas.SizeY/3;
 // chassis
 if (KnockedOut == true)
  {
   Canvas.DrawColor.R = 255;
   Canvas.DrawColor.G = 0;
  }
 else if ((Left_Leg.LegState == L_Falling) && (Right_Leg.LegState == L_Falling))
  {
   Canvas.DrawColor.R = 255;
   Canvas.DrawColor.G = 255;
  }
 else
  {
   Canvas.DrawColor.R = 0;
   Canvas.DrawColor.G = 255;
  }
 Canvas.SetPos(CentXStart, CentYStart);
 Canvas.DrawTile(Texture'DC_Walker_T.HUD.Base', Canvas.SizeY/3, Canvas.SizeY/3, 0.0, 0.0, 512, 512);
 // left leg
 if (WalkerState != W_Walking)
  {
   Canvas.DrawColor.R = 255;
   Canvas.DrawColor.G = 255;
  }
 else if (Left_Leg.LegState == L_OnGround)
  {
   Canvas.DrawColor.R = 0;
   Canvas.DrawColor.G = 255;
  }
 else if (Left_Leg.LegState == L_Interpolating)
  {
   Canvas.DrawColor.R = 255;
   Canvas.DrawColor.G = 255;
  }
 else if (Left_Leg.LegState == L_Falling)
  {
   Canvas.DrawColor.R = 255;
   Canvas.DrawColor.G = 0;
  }
 Canvas.SetPos(CentXStart, CentYStart);
 Canvas.DrawTile(Texture'DC_Walker_T.HUD.Left_Leg', Canvas.SizeY/3, Canvas.SizeY/3, 0.0, 0.0, 512, 512);
 // right leg
 if (WalkerState != W_Walking)
  {
   Canvas.DrawColor.R = 255;
   Canvas.DrawColor.G = 255;
  }
 else if (Right_Leg.LegState == L_OnGround)
  {
   Canvas.DrawColor.R = 0;
   Canvas.DrawColor.G = 255;
  }
 else if (Right_Leg.LegState == L_Interpolating)
  {
   Canvas.DrawColor.R = 255;
   Canvas.DrawColor.G = 255;
  }
 else if (Right_Leg.LegState == L_Falling)
  {
   Canvas.DrawColor.R = 255;
   Canvas.DrawColor.G = 0;
  }
 Canvas.SetPos(CentXStart, CentYStart);
 Canvas.DrawTile(Texture'DC_Walker_T.HUD.Right_Leg', Canvas.SizeY/3, Canvas.SizeY/3, 0.0, 0.0, 512, 512);
 // rockets
 for (i=0; i<20; i++)
  {
   Canvas.SetPos(CentXStart+Rocket_Offsets[i].X*Canvas.SizeY/1536-8, CentYStart+Rocket_Offsets[i].Y*Canvas.SizeY/1536-8);
   if (Rocket_Sensor[i] == 2)
    {
     Canvas.DrawColor.R = 0;
     Canvas.DrawColor.G = 255;
     Canvas.DrawTile(Texture'DC_Walker_T.HUD.Rocket', 16, 16, 0.0, 0.0, 16, 16);
    }
   else if (Rocket_Sensor[i] == 1)
    {
     Canvas.DrawColor.R = 255;
     Canvas.DrawColor.G = 0;
     Canvas.DrawTile(Texture'DC_Walker_T.HUD.Rocket', 16, 16, 0.0, 0.0, 16, 16);
    }
  }
 // check targets
 if (CurrentTarget != none)
  {
   if(HaveRockets() == true)
    {
     Canvas.DrawColor.R = 0;
     Canvas.DrawColor.G = 255;
    }
   else
    {
     Canvas.DrawColor.R = 255;
     Canvas.DrawColor.G = 0;
    }
   Canvas.Font = class'HudBase'.static.GetConsoleFont(Canvas);
   ScreenPos = Canvas.WorldToScreen(CurrentTarget.Location);
   TestOffSet = CurrentTarget.Location;
   TestOffSet.Z =  TestOffSet.Z + CurrentTarget.CollisionHeight;
   HudDistK = VSize(ScreenPos - Canvas.WorldToScreen(TestOffSet))/VSize(CurrentTarget.Location - TestOffSet);
   BoxSize.X = CurrentTarget.CollisionRadius*2*HudDistK + 16;
   BoxSize.Y = CurrentTarget.CollisionHeight*2*HudDistK + 16;
   Canvas.SetPos(ScreenPos.X- BoxSize.X/2, ScreenPos.Y-BoxSize.Y/2);
   Canvas.DrawBox(Canvas, BoxSize.X, BoxSize.Y);
   Canvas.SetPos(ScreenPos.X - BoxSize.X/2 + 4, ScreenPos.Y - BoxSize.Y/2 + 4);
   for(Test_Health = CurrentTarget.Health; Test_Health > 0; Test_Health = Test_Health - 300)
    RocketsCount = RocketsCount + 1;
   Canvas.DrawText("X"$RocketsCount);
  }

 Super.DrawHUD(Canvas);
}

simulated function bool HaveRockets()
 {
  local byte i;

  // find able rocket
  for (i=0; i<20; i++)
   {
    if ((Rockets[i] != none) && (Rockets[i].launched != true))
     return true;
   }
  return false;
 }

simulated function CheckForRockets()
 {
  local byte i;
  for (i=0; i<20; i++)
   {
    if ((Rocket_Sensor[i] == 1) && (Rockets[i].launched != true))
     {
      DetachFromBone(Rockets[i]);
      if (CurrentTarget != none)
       Rockets[i].Seeking = CurrentTarget;
      Rockets[i].Launch();
     }
   }
 }


simulated function CheckForTarget()
 {
  local Vehicle MayBeThis, BestTarget;
  local float   Distance, LastDistance;

  if (PlayerController(Controller) != none)
   {
    Foreach DynamicActors(class'Vehicle', MayBeThis)
     {
      if ( (MayBeThis==Self) || (MayBeThis.Health < 1) || MayBeThis.bDeleteMe || MayBeThis.GetTeamNum() == Team || !MayBeThis.IndependentVehicle() )
       continue;

      if (BestTarget == none)
       BestTarget = MayBeThis;
      else
       {
        Distance = VSize(vector(Controller.Rotation) - Normal(MayBeThis.Location - Location));
        LastDistance = VSize(vector(Controller.Rotation) - Normal(BestTarget.Location - Location));
        if (Distance < LastDistance)
         BestTarget = MayBeThis;
       }
     }
   }
  else if (Bot(Controller) != none)
   {
    if ((Bot(Controller).Enemy != none) && (Bot(Controller).Enemy.IsA('Vehicle')))
     BestTarget = Vehicle(Bot(Controller).Enemy);
   }

   if (BestTarget != none)
    {
     LastDistance = VSize(vector(Controller.Rotation) - Normal(BestTarget.Location - Location));
     if ( (LastDistance < 1) && ((FastTrace(BestTarget.Location, Location) == true) || (Bot(Controller) != none)) )
      CurrentTarget = BestTarget;
     else
      CurrentTarget = none;
    }
   else
    CurrentTarget = none;
 }

simulated function LaunchRocket()
 {
  local byte i;
  local DC_Rocket Launched;

  // find nex able rocket
  for (i=0; i<20; i++)
   {
    Launched = Rockets[Barrel_Order[i]];
    if ((Launched != none) && (Launched.launched != true))
     {
      DetachFromBone(Launched);
      Rocket_Sensor[Launched.RocketIndex] = 1;
      if (CurrentTarget != none)
       Launched.Seeking = CurrentTarget;
      Launched.Launch();
      break;
     }
   }
 }

defaultproperties
{
  /// own physics here ///
  Parked_Upright=30000
  Walking_Upright=40000
  Falling_Upright=2000
  Rotating_Speed=20000
  Balancing_force=200
  Standing_force=500
  StandHeight=350
  SitHeight=150
  RiseSpeed=150
  OnGroundDelay=0.05
  StepSize=220
  CroachSize=192
  WalkerState = W_Initialising
  KnockDowntime = 2
  SpawnHeight = 250
  /// Sounds ///
  IdleSound=Sound'DC_Walker_S.servo_loop'
  StartUpSound=Sound'DC_Walker_S.startup_xeng'
  ShutDownSound=Sound'DC_Walker_S.enddown_xeng'
  SoundVolume=128
  SoundRadius=600
  TransientSoundRadius=1200
  /// Guns ///
  bHasAltFire = true
  bSeparateTurretFocus=true
  DriverWeapons(0)=(WeaponClass=Class'DC_Walker.Quad_gun',WeaponBone="guns_base_bone")
  PassengerWeapons(0)=(WeaponPawnClass=Class'DC_Walker.Top_Turret_Pawn',WeaponBone="turret_base_bone")
  PassengerWeapons(1)=(WeaponPawnClass=Class'DC_Walker.Bottom_Turret_Pawn')
  Barrel(0)="rocket_base_0"
  Barrel(1)="rocket_base_1"
  Barrel(2)="rocket_base_2"
  Barrel(3)="rocket_base_3"
  Barrel(4)="rocket_base_4"
  Barrel(5)="rocket_base_5"
  Barrel(6)="rocket_base_6"
  Barrel(7)="rocket_base_7"
  Barrel(8)="rocket_base_8"
  Barrel(9)="rocket_base_9"
  Barrel(10)="rocket_base_10"
  Barrel(11)="rocket_base_11"
  Barrel(12)="rocket_base_12"
  Barrel(13)="rocket_base_13"
  Barrel(14)="rocket_base_14"
  Barrel(15)="rocket_base_15"
  Barrel(16)="rocket_base_16"
  Barrel(17)="rocket_base_17"
  Barrel(18)="rocket_base_18"
  Barrel(19)="rocket_base_19"
  ///////////
  Barrel_Order(0)=0
  Barrel_Order(1)=14
  Barrel_Order(2)=1
  Barrel_Order(3)=13
  Barrel_Order(4)=2
  Barrel_Order(5)=12
  Barrel_Order(6)=3
  Barrel_Order(7)=11
  Barrel_Order(8)=4
  Barrel_Order(9)=10
  Barrel_Order(10)=5
  Barrel_Order(11)=19
  Barrel_Order(12)=6
  Barrel_Order(13)=18
  Barrel_Order(14)=7
  Barrel_Order(15)=17
  Barrel_Order(16)=8
  Barrel_Order(17)=16
  Barrel_Order(18)=9
  Barrel_Order(19)=15
  /// HUD ///
  Rocket_Offsets(0)=(X=462,Y=82)
  Rocket_Offsets(1)=(X=442,Y=71)
  Rocket_Offsets(2)=(X=422,Y=82)
  Rocket_Offsets(3)=(X=402,Y=71)
  Rocket_Offsets(4)=(X=381,Y=82)
  Rocket_Offsets(5)=(X=462,Y=106)
  Rocket_Offsets(6)=(X=442,Y=117)
  Rocket_Offsets(7)=(X=422,Y=106)
  Rocket_Offsets(8)=(X=402,Y=117)
  Rocket_Offsets(9)=(X=381,Y=106)
  Rocket_Offsets(10)=(X=130,Y=82)
  Rocket_Offsets(11)=(X=110,Y=71)
  Rocket_Offsets(12)=(X=90,Y=82)
  Rocket_Offsets(13)=(X=70,Y=71)
  Rocket_Offsets(14)=(X=50,Y=82)
  Rocket_Offsets(15)=(X=130,Y=106)
  Rocket_Offsets(16)=(X=110,Y=117)
  Rocket_Offsets(17)=(X=90,Y=106)
  Rocket_Offsets(18)=(X=70,Y=117)
  Rocket_Offsets(19)=(X=50,Y=106)
  //////////////////////////////////
  bNetNotify=true
  bAlwaysRelevant=true
  NetUpdateFrequency=30
  NavigationPointRange=64
  CollisionRadius=320.000000
  CollisionHeight=128
  ColLocation=(Z=-450)
  bFollowLookDir=true
  bCanHover=true
  bCanFly=false
  bCanStrafe=true
  bCanWalk=true
  bTurnInPlace=true
  bScriptedRise=false
  bCanBeBaseForPawns=true
  MaxDesireability=0.800000
  DriverDamageMult=0.000000
  ShadowCullDistance=8000.000000
  RedSkin=Shader'DC_Walker_T.Red_Skin'
  BlueSkin=Shader'DC_Walker_T.Blue_Skin'
  HornSounds(0)=Sound'ONSVehicleSounds-S.Horns.LevHorn02'
  DestructionEffectClass=Class'Onslaught.ONSVehDeathRV'
  DamagedEffectClass=none
  UprightStiffness=30000
  UprightDamping=300.000000
  MaxThrustForce=100.000000
  LongDamping=0.010000
  MaxStrafeForce=0.000000
  LatDamping=0.100000
  TurnTorqueFactor=0.000000
  TurnTorqueMax=0.000000
  TurnDamping=300.000000
  MaxYawRate=3.000000
  PitchTorqueFactor=0.000000
  PitchTorqueMax=0.000000
  PitchDamping=20.000000
  RollTorqueTurnFactor=0.000000
  RollTorqueStrafeFactor=0.000000
  RollTorqueMax=0.00000
  RollDamping=30.000000
  StopThreshold=3000.000000
  DrivePos=(X=50,Z=60.000000)
  EntryRadius=380
  TPCamDistRange=(Min=600.000000,Max=1500.000000)
  FPCamPos=(X=0.000000,Z=150.000000)
  TPCamLookat=(X=50.000000,Z=0.000000)
  TPCamWorldOffset=(Z=200.000000)
  bFPNoZFromCameraPitch=true
  bDrawMeshInFP=true
  VehiclePositionString="in a Walker"
  VehicleNameString="Walking Death"
  GroundSpeed=300
  HealthMax=2500.000000
  Health=2500
  Mesh=SkeletalMesh'DC_Walker_A.Chassis'
  VehicleMass=6.000000
  Begin Object Class=KarmaParamsRBFull Name=KParams0
      KInertiaTensor(0)=1.300000
      KInertiaTensor(3)=4.000000
      KInertiaTensor(5)=4.500000
      KLinearDamping=1.0
      KAngularDamping=1.0
      bKStayUpright=True
      KStartEnabled=True
      bHighDetailOnly=False
      bClientOnly=False
      bKDoubleTickRate=True
      bKAllowRotate=True
      bDestroyOnWorldPenetrate=False
      bDoSafetime=True
      KFriction=1.000000
      KImpactThreshold=1000.000000
  End Object
  KParams=KarmaParamsRBFull'DC_Walker.DC_Walker.KParams0'
  bTraceWater=false
}
