class RagUtils extends Object;

/**
 *  Quick Summary of RagUtils Methods:
 *
 *  To Move Ragdolls.
 *  - PullRagdollToLocation
 *  - SetRagdollVelocity (approximate)
 *
 *  To keep Unreal from deresing or destroying a ragdoll.
 *  - RagdollKWake
 *  - ResetLastRenderTime
 *  - ResetRagdollLifeSpan
 *  - SetKImportance
 *
 *  Misc.
 *  - IsARagdoll
 *  - GetCappedVector
 *  - LocationBehindActor
 *  - MagicVector
 *  - PointAlongLine
 *  - Distance
 */

//Random note: Can make invisible ragdoll with setbonescale, i think.



/**
 *  Just a bunch of nice ragdoll-related static methods.
 *  You can call these methods from anywhere in your code.
 *
 *  If I use an Actor instead of a xPawn anywhere, it is only
 *  because that method supposidly works on Actors too.
 *  I do not actually know what happens if you use the method on
 *  a different Actor than an xPawn ragdoll.
 *  Maybe you will find some use for that?
 */

/**
 *  Several of my ragdoll movement methods have versions where
 *  they use default values from this list of constants.
 *  In some cases you can specify your own values in the
 *  method arguments.
 */
var float DefAcceptableDistance;
var float DefRopeDamping;
var float DefRopeStiffness;
var float DefMaxImpulse;
var name DefImpactBone;

/**
 *  PullRagdollToLocation will use the Actor.KAddimpulse method
 *  to drag a ragdoll in a straight line towards a certain location.
 *  (Specifically, it calls RagUtils.SetRagdollVelocity,
 *   which then uses Actor.KAddimpulse)
 *  Will do nothing if the passed xPawn is not a Ragdoll.
 *  You can call this method with different levels of control,
 *  depending on what method version you call.
 *
 *  Parameters:
 *   Ragdoll - The ragdoll to pull
 *   Target - The x,y,z location to pull ragdoll to
 *   AcceptableDistance - How close to the target is "close enough" that we
 *                        do not really need to pull the ragdoll right now?
 *   MaxImpulse - What is the maximum impulse that will
 *                be allowed to be applied to a ragdoll.
 *                (some ragdolls will fly out the stage
 *                 if the impulse is not capped)
 *   RopeDamping - Not really sure, took it from NinjaRope2 by Jeff Wofford
 *   RopeStiffness - Not really sure, took it from NinjaRope2 by Jeff Wofford
 */
static function PullRagdollToLocation(xPawn Ragdoll, vector Target)
{
    //PullToLocationOld(Ragdoll,Target);
    PullRagdollToLocationD(Ragdoll,Target,
                          default.DefAcceptableDistance);
}
static function PullRagdollToLocationD(xPawn Ragdoll, vector Target,
                               float AcceptableDistance)
{
    PullRagdollToLocationDM(Ragdoll,Target,
                          AcceptableDistance,
                          default.DefMaxImpulse);
}
static function PullRagdollToLocationDM(xPawn Ragdoll, vector Target,
                               float AcceptableDistance,
                               float MaxImpulse)
{
    PullRagdollToLocationDMR(Ragdoll,Target,
                          AcceptableDistance,
                          MaxImpulse,
                          default.DefRopeDamping, default.DefRopeStiffness);
}
static function PullRagdollToLocationDMR(xPawn Ragdoll, vector Target,
                               float AcceptableDistance,
                               float MaxImpulse,
                               float RopeDamping, float RopeStiffness)
{
    local vector DisplacementToTarget, DesiredVelocity, DirectionToTarget;
    local float CurrentDistanceToTarget;
    if(!IsARagdoll(Ragdoll))
        return;

    //DisplacementToTarget is a vector from follower >-to-> target
    DisplacementToTarget = Target - Ragdoll.Location;
    CurrentDistanceToTarget = VSize(DisplacementToTarget);
    // Is pulling needed?
    if( AcceptableDistance < CurrentDistanceToTarget )
    {
        //Get a normal (1 unit in size) DirectionToTarget vector
        DirectionToTarget = DisplacementToTarget / CurrentDistanceToTarget;

        // Hook's spring law??? not anymore probably...
        // I took this from the ninja rope code created by Jeff Wofford.
        //DesiredVelocity = -DirectionToTarget *
        //    ( RopeStiffness * ( AcceptableDistance - CurrentDistance ) -
        //    RopeDamping * ( Ragdoll.Velocity dot DirectionToTarget ));

        // Dot product = How much are these vecs going in the same direction?
        // Dot product resulting from 2 vectors:
        // Perpendicular vecs, goes to zero
        // Same direction vecs, get a big positive number
        // Diff direction vecs, get a big negative number
        //
        // Basically
        // RopeDamping says: do not approach too fast
        // But do not quickly fly away either
        DesiredVelocity = DirectionToTarget *
            ( RopeStiffness * ( CurrentDistanceToTarget - AcceptableDistance));
            //+ RopeDamping * (Ragdoll.Velocity dot force));

        SetRagdollVelocityM(Ragdoll,DesiredVelocity,MaxImpulse);
    }
}

/**
 *  SetRagdollVelocity allows you to move a ragdoll at a specific velocity.
 *  It uses Actor.KAddimpulse to do this.
 *  It will do nothing if the passed xPawn is not a ragdoll.
 *
 *  Parameters:
 *   Ragdoll - Ragdoll to set velocity for
 *   Velocity - Velocity to apply to Ragdoll
 *   MaxImpulse - The maximum impulse that will be applied to the Ragdoll
 *                (some ragdolls will fly out the stage
 *                 if the impulse is not capped)
 *   ImpactBone - Name of the bone to apply the force to.
 *                Use at your own risk, I have only tested 'spine'
 */
static function SetRagdollVelocity(xPawn Ragdoll, vector Velocity)
{
    SetRagdollVelocityM(Ragdoll,Velocity,
                       default.DefMaxImpulse);
}
static function SetRagdollVelocityM(xPawn Ragdoll, vector Velocity,
                            float MaxImpulse)
{
    SetRagdollVelocityMB(Ragdoll, Velocity,
                       MaxImpulse,
                       default.DefImpactBone);
}
static function SetRagdollVelocityB(xPawn Ragdoll, vector Velocity,
                            name ImpactBone)
{
    SetRagdollVelocityMB(Ragdoll, Velocity,
                       default.DefMaxImpulse,
                       ImpactBone);
}
static function SetRagdollVelocityMB(xPawn Ragdoll, vector Velocity,
                            float MaxImpulse,
                            name ImpactBone)
{
    local vector Impulse, VelocityToApply;
    //local float lindamp, angdamp;
    //local float dummy;
    if(!IsARagdoll(Ragdoll))
        return;
    //First we "neutralize" the existing velocity (-Ragdoll.Velocity)
    VelocityToApply = Velocity - Ragdoll.Velocity;
    //Then multiply by the mass to translate velocity into "impulse"
    Impulse = VelocityToApply * Ragdoll.Mass * Ragdoll.KGetMass() * 0.7;
    /*
    //It seems that all ragdolls have Mass == 100 and KGetMass() == 1
    //Yet some experience a lot stronger influcence from the
    //KAddImpulse then other ragdolls
    //(A lot of custom girl radolls and Agent Smith i've noticed)
    //Mass=100.00
    //KGetMass=1.00
    //ScriptLog: LinDamp = 0.15
    //ScriptLog: AngDamp = 0.05
    //KFriction = 0.60
    Ragdoll.KGetDampingProps(lindamp,angdamp);
    if(Ragdoll.Mass != 100.00 && Ragdoll.KGetMass() != 1.00
    && lindamp != 0.15 && AngDamp != 0.05 && Ragdoll.KGetFriction() != 0.60)
    {
        //Ragdolls don't get to keep their pri's of course.
        //So you cannot get their names, as far as I know.
        Log("LinDamp = " $ lindamp);
        Log("AngDamp = " $ angdamp);
        Log(" Mass = " $ Ragdoll.Mass);
        Log("KMass = " $ Ragdoll.KGetMass());
        Log("KFriction = " $ Ragdoll.KGetFriction());
    }*/

    //Log("Desired Impulse = " $ vsize(Impulse));

    //Cap the impulse magnitude or ragdolls get lost (they fly out the stage)
    //Some ragdolls are worse than others
    Impulse = GetCappedVector(Impulse,MaxImpulse);
    //Log("OldBone = " $ ImpactBone);
    //Log("ImpulseSize = " $ vsize(Impulse));
    //ImpactBone = Ragdoll.GetClosestBone(Ragdoll.Location,//Where to start looking for the bone?
    //                                             Normal(Impulse),//Direction to look in?
    //                                             Dummy);//out Distance had to look to find the bone.
    //Log("NewBone = " $ ImpactBone);
    //Apply the impulse
    Ragdoll.KAddimpulse(Impulse,
                        Ragdoll.Location,//vect(0,0,0)
                        ImpactBone);
}

/**
 *  IsARagdoll is just a quick check.
 *  I would like to change it to: "Can treat like a ragdoll" or something,
 *  so that these methods could be called on actors other than xPawns.
 *  Maybe it already does that, I just need to change the argument from
 *  xPawn Ragdoll ->to-> Actor Ragdoll.
 *  I dunno.
 *  Please contact me if you know if I can safely do this,
 *  or if there is another way I should do this.  I will change this code.
 */
static function bool IsARagdoll(xPawn Ragdoll)
{
    return (
             (
               Ragdoll != None
             )
           &&
             (
                Ragdoll.Physics == PHYS_KarmaRagDoll
             || Ragdoll.Physics == PHYS_Karma
             )
           );
}

/**
 *  Probably should move into a VectUtil class
 *  It only concerns vectors.
 *  Still, it is used by SetRagdollVelocity.
 */
static function vector GetCappedVector(Vector v, float maxvsize)
{
    local float vecsize;
    vecsize = VSize(v);
    maxvsize = Abs(maxvsize);

    if(vecsize > maxvsize)
        return v*maxvsize/vecsize;
    else
        return v;
}

/**
 *  Vector methods used by ChainPuller classes
 *  - LocationBehindActor
 *  - MagicVector
 *  - Distance
 */
static function vector LocationBehindActor(Actor A, float Distance)
{
    local rotator rota;
    local Pawn P;
    P = Pawn(A);
    if(P != None)
        rota = P.GetViewRotation();
    else
        rota = A.Rotation;
    //This is a location behind where the actor is facing
    return A.Location - normal(vector(rota))*Distance;
}
// I'm not really sure how to describe or what to call this vector.
static function vector MagicVector(Actor A1, Actor A2, float Distance)
{
    return MagicVectorV(A1.Location,A2.Location,Distance);
}
static function vector MagicVectorV(Vector V1, Vector V2, float Distance)
{
    return V1 - normal(V2-V1)*Distance;
}
static function vector PointAlongLine(Vector PivotPoint, Vector V2, float Distance)
{
    return PivotPoint + Distance*Normal(V2 - PivotPoint);
}
static function float Distance(Vector V1, Vector V2)
{
    return VSize(V1-V2);
}

/**
 *  The Following methods are used to keep Unreal
 *  from deresing or destroying a ragdoll.
 *  - RagdollKWake (WARNING: don't use this one, the rest are ok though)
 *  - ResetLastRenderTime
 *  - ResetRagdollLifeSpan
 *  - SetKImportance
 *  A lot of the reasoning for this code can be found in xPawn.uc
 *  But I'll give you a quick summary of what I remember for each
 */

/**
 * WARNING:
 * I'm pretty sure that calling RagdollKWake a lot will
 * make the ragdoll telefrag players.
 * I no longer use this method.  I don't believe it is needed anyway.
 * KAddImpulse seems to wake them up pretty good enough.
 *
 * I THINK the reason for RagdollKWake is:
 *   Sometime a ragdoll that is not moving goes to sleep
 *   A sleeping ragdoll has no more ragdoll physics
 *   It's joints don't bend.  It acts like it has rigamortis or something
 * I THINK that if there are too many ragdolls, they go to sleep, but
 *   even this method will have no effect. But call it anyway.
 */
/*
static function RagdollKWake(Actor Ragdoll)
{
    if(!Ragdoll.KIsAwake())
        Ragdoll.KWake();
}
*/

/**
 * I use ResetLastRenderTime to keep Unreal from destroying ragdolls
 *   Unreal will destroy ragdolls that have not been rendered in a while
 *   It keeps track of this with Ragdoll.LastRenderTime
 *   So I trick it, by making it think that the Ragdoll was just rendered.
 */
static function ResetLastRenderTime(Actor Ragdoll)
{
    Ragdoll.LastRenderTime = Ragdoll.Level.TimeSeconds;
}

/**
 * I use ResetRagdollLifeSpan to keep Unreal from deresing ragdolls.
 *   Unreal will deres a ragdoll when its LifeSpan gets too small.
 *   LifeSpan ticks down on every frame.
 *     (Also, there are times where Unreal will reset a Ragdoll's LifeSpan
 *      to a lower value. See: xPawn. State Dying event KVelDropBelow())
 *   I think they normally deres at (LifeSpan < 6)
 */
static function ResetRagdollLifeSpan(xPawn Ragdoll)
{
    Ragdoll.LifeSpan = Ragdoll.RagdollLifeSpan;
}

/**
 * I use SetKImportance(Ragdoll, true) to encourage Unreal to keep ragdolls around
 * From KarmaParamsSkel.uc:
 *  - "This indicates this ragdoll will not
 *     be recycled during KMakeRagdollAvailable"
 * Basically, I think this means that:
 *  There is a max number of ragdolls
 *    (you can set this with code Level.MaxRagdolls = 999 <or whatever number>)
 *  When that max is reached, if a new Ragdoll is introduced (someone died),
 *    Unreal will destroy a random ragdoll with (bKImportance == false)
 *
 *  I know bad things can happen if you just use all these methods to make
 *   all ragdolls last forever with a huge MaxRagdolls limit..
 *   but I forgot the specifics.
 *  I kinda remember something like this happening:
 *
 *  If the max is reached and ALL ragdolls are bKImprtance == true,
 *  I think Unreal will start freezing ragdolls,
 *  or maybe no new ragdolls are created.
 *
 *    (Note: even if you set a very high Level.MaxRagdolls,
 *           at some point Unreal will have problems if you have too many.
 *            I think it will start freezing ragdolls if there are too many,
 *            but maybe this is only when all ragdolls have bKImportance == true,
 *            I cannot remember.)
 */
static function SetKImportance(Actor Ragdoll, bool KImportance)
{   //Also keeps the ragdoll from disappearing
    local KarmaParamsSkel kskel;
    kskel = KarmaParamsSkel(Ragdoll.KParams);
    if(kskel != None)
        kskel.bKImportantRagdoll = KImportance;
}

static function bool GetKImportance(Actor Ragdoll)
{
    local KarmaParamsSkel kskel;
    if(Ragdoll == None) return false;

    kskel = KarmaParamsSkel(Ragdoll.KParams);
    if(kskel != None)
        return kskel.bKImportantRagdoll;
    else
        return false;
}

//
// This was what I used a long time ago.
// It hasn't really changed much.
//
/*
static function PullToLocationOld(xPawn Ragdoll, vector Target)
{
    local vector force, RopePullMomentum;
    local float CurrentDistance, PullMagnitude;
    if(Ragdoll.Physics != PHYS_Karma && Ragdoll.Physics != PHYS_KarmaRagDoll)
        return;

    //force is a vector from follower >-to-> target
    force = Target - Ragdoll.Location;
    CurrentDistance = VSize(force);
    // Is pulling needed?
    if( default.DefAcceptableDistance < CurrentDistance )
    {
        //Normalize force vector
        force /= CurrentDistance;

        // Hook's spring law
        //At this point, RopePullMomentum is the desired velocity for the ragdoll
        RopePullMomentum = -force *
            ( default.DefRopeStiffness *
             ( default.DefAcceptableDistance - CurrentDistance ) -
            default.DefRopeDamping *
             ( Ragdoll.Velocity dot force ));

        //Here we "neutralize" the existing velocity (-Ragdoll.Velocity)
        //Them multiply by the mass to translate velocity into "impulse"
        RopePullMomentum = ( RopePullMomentum - Ragdoll.Velocity )
                           * Ragdoll.Mass * Ragdoll.KGetMass();
        //Cap the impulse magnitude or ragdolls get lost
        PullMagnitude = VSize(RopePullMomentum);
        if(PullMagnitude > 85000.00f)
            RopePullMomentum *= 85000.00f/PullMagnitude;
        Ragdoll.KAddimpulse(RopePullMomentum,
                            Ragdoll.Location,
                            'spine');


        }
}
*/

//Pawns have RootBone, HeadBone, SpineBone1, SpineBone2 vars

      //RagdollSeparationDistance=30.000000
      //bKImportantRagdoll=false
      //bTearOff=true
defaultproperties
{
    DefAcceptableDistance = 13.0f
    DefRopeDamping        = 0.3f
    DefRopeStiffness      = 3.0f
    DefMaxImpulse         = 50000.0f
    DefImpactBone         = spine
}
