/* 
    BattleTeamArena Copyright (C) 2007-2008 Nico de Vries.

    This file is part of BattleTeamArena.

    BattleTeamArena is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see http://www.gnu.org/licenses/.
    
    This file also contains code that is derived from code which is
    Copyright 1998-2007 Epic Games, Inc. All Rights Reserved. That part 
    of the code is covered by the Epic license and not by the GPL.
*/

class BattlePlayerController extends UTPlayerController;

var globalconfig int HitSoundSetting;
var globalconfig int SkinSetting;
var globalconfig int HUDSetting;

var array<soundCue> HitSoundSounds;
var() ESpecialPlayerState ClientSpecialPlayerState;
var BattleInteraction BI;
var string CamMode;
var BattleDynamicMenu BDM;

var bool bZoomActive;
var float LastFOVAngle;
var int TickCtr;

var bool bProtectFOV;
var float FOVBackup;

simulated function ProtectFOV ()
{
  bProtectFOV = true;
  BackupFOV ();
}

simulated function UnprotectFOV ()
{
  bProtectFOV = false;
}

simulated function BackupFOV ()
{
  FOVBackup = FOVAngle;
}

simulated function RestoreFOV ()
{
  if (bProtectFOV) {
    if (FOVAngle != FOVBackup) {
      FOVAngle = FOVBackup;
    }
  }
}

simulated function StartZoom(float NewDesiredFOV, float NewZoomRate)
{
  bZoomActive = true;
  Super.StartZoom (NewDesiredFOV, NewZoomRate);
  ProtectFOV ();
}

simulated function EndZoom()
{
  UnprotectFOV ();
  Super.EndZoom ();
  bZoomActive = false;
}

function AdjustFOV(float DeltaTime)
{
  RestoreFOV();
  Super.AdjustFOV (DeltaTime);
  BackupFOV();
}

event float GetFOVAngle()
{
  if (PlayerCamera != None) {
    return PlayerCamera.GetFOVAngle();
  } else {
    RestoreFOV();
    return FOVAngle;
  }
}

reliable server function ServerShowMessage (int Switch, optional PlayerReplicationInfo RelatedPRI_1, optional PlayerReplicationInfo RelatedPRI_2)
{
  ClientShowMessage (Switch, RelatedPRI_1, RelatedPRI_2);
}

reliable client function ClientShowMessage (int Switch, optional PlayerReplicationInfo RelatedPRI_1, optional PlayerReplicationInfo RelatedPRI_2)
{
  ReceiveLocalizedMessage (class'BattleMessage', Switch, RelatedPRI_1, RelatedPRI_2);
}

function GameHasEnded(optional Actor EndGameFocus, optional bool bIsWinner)
{
  SetViewTarget (EndGameFocus);
  GotoState ('RoundEnded');
  ClientGameEnded2 (EndGameFocus, bIsWinner, EndGameFocus.Rotation);
}

reliable server function AdminSetCampingSetting (int Value)
{
  if (PlayerReplicationInfo.bAdmin || WorldInfo.NetMode == NM_Standalone) { // E.g. in case someone hacked the code on the client
    BattleGRI (WorldInfo.GRI).CampingSetting = Value;
    BattleGRI (WorldInfo.GRI).SaveConfig();
  }
}

reliable server function AdminSetChallengeModeSetting (int Value)
{
  if (PlayerReplicationInfo.bAdmin || WorldInfo.NetMode == NM_Standalone) { // E.g. in case someone hacked the code on the client
    BattleGRI (WorldInfo.GRI).ChallengeModeSetting = Value;
    BattleGRI (WorldInfo.GRI).SaveConfig();
  }
}

reliable server function AdminSetRoundLengthSetting (int Value)
{
  if (PlayerReplicationInfo.bAdmin || WorldInfo.NetMode == NM_Standalone) { // E.g. in case someone hacked the code on the client
    BattleGRI (WorldInfo.GRI).RoundLengthSetting = Value;
    BattleGRI (WorldInfo.GRI).SaveConfig();
  }
}

reliable server function AdminSetHUDSetting (int Value)
{
  if (PlayerReplicationInfo.bAdmin || WorldInfo.NetMode == NM_Standalone) { // E.g. in case someone hacked the code on the client
    BattleGRI (WorldInfo.GRI).HUDSetting = Value;
    BattleGRI (WorldInfo.GRI).SaveConfig();
  }
}

reliable server function AdminSetFovSetting (int Value)
{
  if (PlayerReplicationInfo.bAdmin || WorldInfo.NetMode == NM_Standalone) { // E.g. in case someone hacked the code on the client
    BattleGRI (WorldInfo.GRI).FOVSetting = Value;
    BattleGRI (WorldInfo.GRI).SaveConfig();
  }
}

reliable server function AdminSetRemoveSetting (int Value)
{
  if (PlayerReplicationInfo.bAdmin || WorldInfo.NetMode == NM_Standalone) { // E.g. in case someone hacked the code on the client
    BattleGRI (WorldInfo.GRI).RemoveSetting = Value;
    BattleGRI (WorldInfo.GRI).SaveConfig();
    BattleTeamArena(Worldinfo.Game).RemoveStuff();
  }
}

reliable server function AdminSetWeaponsSetting (int Value)
{
  if (PlayerReplicationInfo.bAdmin || WorldInfo.NetMode == NM_Standalone) { // E.g. in case someone hacked the code on the client
    BattleGRI (WorldInfo.GRI).WeaponsSetting = Value;
    BattleGRI (WorldInfo.GRI).SaveConfig();
  }
}

reliable server function AdminSetAmmoSetting (int Value)
{
  if (PlayerReplicationInfo.bAdmin || WorldInfo.NetMode == NM_Standalone) { // E.g. in case someone hacked the code on the client
    BattleGRI (WorldInfo.GRI).AmmoSetting = Value;
    BattleGRI (WorldInfo.GRI).SaveConfig();
  }
}

reliable server function AdminSetImpactHammerSetting (int Value)
{
  if (PlayerReplicationInfo.bAdmin || WorldInfo.NetMode == NM_Standalone) { // E.g. in case someone hacked the code on the client
    BattleGRI (WorldInfo.GRI).ImpactHammerSetting = Value;
    BattleGRI (WorldInfo.GRI).SaveConfig();
  }
}

reliable client function ClientGameEnded2 (Actor EndGameFocus, bool bIsWinner, Rotator TheRot)
{
  Super.ClientGameEnded (EndGameFocus, bIsWinner);
  EndGameFocus.SetRotation (EndGameFocus.Rotation);
  SetRotation (EndGameFocus.Rotation);
}

state PlayerClimbing
{
  function ProcessMove(float DeltaTime, vector NewAccel, eDoubleClickDir DoubleClickMove, rotator DeltaRot)
  {
    if (ClientSpecialPlayerState != SPS_Frozen) {
      Super.ProcessMove(DeltaTime, NewAccel, DoubleClickMove, DeltaRot);
    }
  }

  function ReplicateMove (float DeltaTime, vector NewAccel, eDoubleClickDir DoubleClickMove, rotator DeltaRot)
  {
    if (ClientSpecialPlayerState != SPS_Frozen) {
      Super.ReplicateMove(DeltaTime, NewAccel, DoubleClickMove, DeltaRot);
    }
  }
}

state PlayerWalking
{
  function ProcessMove(float DeltaTime, vector NewAccel, eDoubleClickDir DoubleClickMove, rotator DeltaRot)
  {
    if (ClientSpecialPlayerState != SPS_Frozen) {
      Super.ProcessMove(DeltaTime, NewAccel, DoubleClickMove, DeltaRot);
    }
  }

  function ReplicateMove (float DeltaTime, vector NewAccel, eDoubleClickDir DoubleClickMove, rotator DeltaRot)
  {
    if (ClientSpecialPlayerState != SPS_Frozen) {
      Super.ReplicateMove(DeltaTime, NewAccel, DoubleClickMove, DeltaRot);
    }
  }
}

state PlayerSwimming
{
  function ProcessMove(float DeltaTime, vector NewAccel, eDoubleClickDir DoubleClickMove, rotator DeltaRot)
  {
    if (ClientSpecialPlayerState != SPS_Frozen) {
      Super.ProcessMove(DeltaTime, NewAccel, DoubleClickMove, DeltaRot);
    }
  }

  function ReplicateMove (float DeltaTime, vector NewAccel, eDoubleClickDir DoubleClickMove, rotator DeltaRot)
  {
    if (ClientSpecialPlayerState != SPS_Frozen) {
      Super.ReplicateMove(DeltaTime, NewAccel, DoubleClickMove, DeltaRot);
    }
  }
}

state Dead
{
  exec function StartFire (optional byte FireModeNum)
  {
    if (FireModeNum==0) {
      SpectateCycle();
    } else {
      SpectateMe();
    }
  }
}
 
function bool CanSpectate (PlayerReplicationInfo PRI)
{
  if (PRI == PlayerReplicationInfo) return false; // Exclude self from the cycle
  if (!BattleGRI (WorldInfo.GRI).bRoundIsOver) { // You can watch players of the other team between rounds
    if (PRI.Team != PlayerReplicationInfo.Team) return false; // Can only view team mates
  }
  if (Controller(PRI.Owner).Pawn == None) return false; // There must be something to view
  if (Controller(PRI.Owner).IsDead()) return false; // It should be alive, this includes frozen players BTW
  return true;
}

// Stub because the client cannot call BPRI.RequestStatReplication directly
reliable server function RequestStatReplication (BattlePRI BPRI)
{
  BPRI.RequestStatReplication ();
}

function BattlePRI NextInCycle (Actor Current, optional string CycleType="spectate", optional int Direction=1) {
  local int i, CurrentIndex, NewIndex;
  local BattlePRI BPRI;

  // Determine the BPRI of what we are currently looking at
  if (Current == Pawn) {
    BPRI = BattlePRI (PlayerReplicationInfo);
  } else {
    BPRI = BattlePRI (Current);
  }

  // Locate current BPRI;
  CurrentIndex = -1;
  for (i=0; i<WorldInfo.GRI.PRIArray.Length; i++) {
    if (BPRI == WorldInfo.GRI.PRIArray[i]) {
      CurrentIndex = i;
      break;
    }
  }

  // Locate next BPRI
  for (NewIndex=CurrentIndex+Direction; (NewIndex>=0)&&(NewIndex<WorldInfo.GRI.PRIArray.Length); NewIndex=NewIndex+Direction) {
    BPRI = BattlePri (WorldInfo.GRI.PRIArray[NewIndex]);
    if (BPRI != None) {
      if (CycleType == "spectate" && (Controller(BPRI.Owner) != None) && CanSpectate(BPRI) && !BPRI.bOnlySpectator) {
        return BPRI;
      } else if (CycleType == "viewstats" && !BPRI.bOnlySpectator) {
        return BPRI;
      }
	}
  }
  CurrentIndex = (NewIndex < 0) ? WorldInfo.GRI.PRIArray.Length : -1;
  for (NewIndex=CurrentIndex+Direction; (NewIndex>=0)&&(NewIndex<WorldInfo.GRI.PRIArray.Length); NewIndex=NewIndex+Direction) {
    BPRI = BattlePRI (WorldInfo.GRI.PRIArray[NewIndex]);
    if (BPRI != None) {
      if (CycleType == "spectate" && (Controller(BPRI.Owner) != None) && CanSpectate(BPRI) && !BPRI.bOnlySpectator) {
        return BPRI;
      } else if (CycleType == "viewstats" && !BPRI.bOnlySpectator) {
        return BPRI;
      }
    }
  }
  return None; 
}

function ChangeViewpoint (optional string mode="swap")
{
  if (bZoomActive) return;
  ClientEndZoom();
  if (mode=="1st") {
    CamMode = "1st";
    SetCameraMode ('FirstPerson');
    ResetCameraMode();
  } else if (mode=="3rd") {
    CamMode = "3rd";
    SetCameraMode ('ThirdPerson');
    FixFOV();
  } else {
    if (CamMode=="1st") {
      ChangeViewpoint ("3rd");
    } else {
      ChangeViewpoint ("1st");
    }
  }
}

reliable server function SpectateCycle()
{
  local BattlePRI BPRI;
  
  BPRI = NextInCycle (RealViewTarget, "spectate");
  if (BPRI != None && RealViewTarget != BPRI) {
    ServerShowMessage (1, BPRI);
    if (BPRI == PlayerReplicationInfo) {
      if (Pawn != None) {
        SetViewTarget (Pawn);
      } else {
        SetViewTarget (BPRI);
      }
    } else {
      SetViewTarget (BPRI);
    }
  }
}

reliable client function PlayStartupMessage(byte StartupStage)
{
	if ( StartupStage == 7 )
	{
		ReceiveLocalizedMessage( class'UTTimerMessage', 17, PlayerReplicationInfo );
	}
	else
	{   
        ReceiveLocalizedMessage( class'BattleStartupMessage', StartupStage, PlayerReplicationInfo );
	}
}

unreliable client function HitSound (int Damage)
{
  local int Index;
  local bool reverse;

  if (WorldInfo.NetMode==NM_DedicatedServer) return;
  if (HitSoundSetting==999) return;
  Index = HitSoundSetting / 2;
  Reverse = ! (Index * 2 - HitSoundSetting == 0);
  if (Reverse) {
    HitSoundSounds[Index].PitchMultiplier = 1.2 + Min (100, Max (1, 100 - Damage)) / 100.0;
  } else {
    HitSoundSounds[Index].PitchMultiplier = 1.2 + Min (100, Max (1, Damage)) / 100.0;
  }
  HitSoundSounds[Index].VolumeMultiplier = 2.0;
  if(ViewTarget!=None) ViewTarget.PlaySound (HitSoundSounds[Index], true);
}

exec function FOV(float F)
{
  if (bZoomActive) return;
  if ((F >= 80.0) || (WorldInfo.NetMode==NM_Standalone) || PlayerReplicationInfo.bOnlySpectator) {
    OnFootDefaultFOV = FClamp(F, 80, BattleGRI (WorldInfo.GRI).FovSetting);
    if (Vehicle(Pawn) == None) {
      FixFOV();
    }
    SaveConfig();
  }
}

function FixFOV()
{
  if (bZoomActive) return;
  if (OnFootDefaultFOV < 80) {
    OnFootDefaultFOV = 90.0;
  }
  OnFootDefaultFOV = FClamp(OnFootDefaultFOV, 80, BattleGRI (WorldInfo.GRI).FovSetting);
  FOVAngle = OnFootDefaultFOV;
  DesiredFOV = OnFootDefaultFOV;
  DefaultFOV = OnFootDefaultFOV;
}

reliable server function SpectateMe() 
{
  if (Pawn != None) {
    SetViewTarget (Pawn);
  }
}

simulated event PlayerTick (float deltatime)
{
  TickCtr++; 
  if (LastFOVAngle != FOVAngle) {
    LastFOVAngle = FOVAngle;
  }

  if (BDM != None) {
    BDM.TickAnyway ();
  }

  // Handle 3rd person look while frozen
  if (BattlePRI (PlayerReplicationInfo) != None) {
    if (BattlePRI (PlayerReplicationInfo).SpecialPlayerState == SPS_Frozen) {
      if (ClientSpecialPlayerState != SPS_Frozen) {
        ClientSpecialPlayerState = SPS_Frozen;
        ChangeViewpoint ("3rd");
      }
    } else if (BattlePRI (PlayerReplicationInfo).SpecialPlayerState == SPS_Alive) {
      if (ClientSpecialPlayerState != SPS_Alive) {
        ClientSpecialPlayerState = SPS_Alive;
        ChangeViewpoint ("1st");
        SpectateMe ();
      }
    } else if (BattlePRI (PlayerReplicationInfo).SpecialPlayerState == SPS_VeryDead) {
      if (ClientSpecialPlayerState != SPS_VeryDead) {
        ClientSpecialPlayerState = SPS_VeryDead;
      }
    }
  }

  // Create a BattleInteraction and connect it to the BattlePRI  
  if (BI == None) {
    BI = New(Self) class'BattleInteraction';
    Interactions.Insert(0,1);
    Interactions[0] = BI;
    BI.Init();
    BI.PC = Self;
  }

  if (LastFOVAngle != FOVAngle) {
    LastFOVAngle = FOVAngle;
  }
  Super.PlayerTick (DeltaTime);
  if (LastFOVAngle != FOVAngle) {
    LastFOVAngle = FOVAngle;
  }

  if (WorldInfo != None && BattleGRI(WorldInfo.GRI) != None) {
    BattleGRI(WorldInfo.GRI).LocalBattlePlayerController = Self;
  }
  
  if (LastFOVAngle != FOVAngle) {
    LastFOVAngle = FOVAngle;
  }
  if (BattleGRI (WorldInfo.GRI).FovSetting < OnFootDefaultFOV) {
    FOV (BattleGRI (WorldInfo.GRI).FovSetting);
  }
  if (LastFOVAngle != FOVAngle) {
    LastFOVAngle = FOVAngle;
  }
}
 
defaultproperties
{
  InputClass=Class'BattlePlayerInput'
  HitSoundSounds(0)=SoundCue'A_Pickups.Ammo.Cue.A_Pickup_Ammo_Flak_Cue'
  HitSoundSounds(1)=SoundCue'BattleTeamArenaContent.randomcue'
  HitSoundSounds(2)=SoundCue'BattleTeamArenaContent.arrowcue'
  HitSoundSounds(3)=SoundCue'BattleTeamArenaContent.cashcue'
  HitSoundSounds(4)=SoundCue'BattleTeamArenaContent.flickcue'
  HitSoundSounds(5)=SoundCue'BattleTeamArenaContent.pingwarbcue'
  HitSoundSounds(6)=SoundCue'BattleTeamArenaContent.plopcue'
  HitSoundSounds(7)=SoundCue'BattleTeamArenaContent.pockcue'
  HitSoundSounds(8)=SoundCue'BattleTeamArenaContent.poingcue'
  HitSoundSounds(9)=SoundCue'BattleTeamArenaContent.rampcue'
  HitSoundSounds(10)=SoundCue'BattleTeamArenaContent.shotcue'
  HitSoundSounds(11)=SoundCue'BattleTeamArenaContent.tockcue'
  HitSoundSounds(12)=SoundCue'BattleTeamArenaContent.whizpopcue'
  HitSoundSounds(13)=SoundCue'BattleTeamArenaContent.woodcue'
  HitSoundSounds(14)=SoundCue'BattleTeamArenaContent.wooshcue'
  HitSoundSounds(15)=SoundCue'BattleTeamArenaContent.zapcue'
  HitSoundSetting=0;
  SkinSetting=0;
  HUDSetting=0;
  Name="BattlePlayerController"
}