/* 
    BattleTeamArena Copyright (C) 2007-2008 Nico de Vries.

    This file is part of BattleTeamArena.

    BattleTeamArena is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see http://www.gnu.org/licenses/.
*/ 
 
`include (BattleTeamArena/Classes/BattleTeamArena.uci) 

class BattlePRI extends UTPlayerReplicationInfo;

/*
    The stat data replication consists of 5 layers:

    * server side raw stat data (StatsServer)
    * server side raw stat data for the current round (StatsRoundServer)
    * server side raw stat data that should be replicated (StatsServerP4R) "prepared for replication"
    * server+client side raw stat data that is actually replicated (Stats)
    * client side polished data that can be presented (StatsPres)

    All layers have dirty counters to minimize network and CPU overhead. The clients will send update 
    requests for the data they are actually viewing to the server, so only data visible on some client 
    is really replicated. The 2 separate layers on the server are there to make sure the actual 
    replication overhead is determined by the server and not by the client requests.
*/

// Server side data
var int TickCtr_Fast;
var int TickCtr_Slow;
var bool bDoomed;
var int StatsServer_Shots[`STATS_Types];
var int StatsServer_HitsTo[`STATS_Types];
var int StatsServer_HitsBy[`STATS_Types];
var int StatsServer_DamageTo[`STATS_Types];
var int StatsServer_DamageBy[`STATS_Types];
var int StatsServer_KilledTo[`STATS_Types];
var int StatsServer_KilledBy[`STATS_Types];
var int StatsServer_DirtyCTR; 
var int StatsRoundServer_Shots[`STATS_Types];
var int StatsRoundServer_HitsTo[`STATS_Types];
var int StatsRoundServer_HitsBy[`STATS_Types];
var int StatsRoundServer_DamageTo[`STATS_Types];
var int StatsRoundServer_DamageBy[`STATS_Types];
var int StatsRoundServer_KilledTo[`STATS_Types];
var int StatsRoundServer_KilledBy[`STATS_Types];
var int StatsRoundServer_Thawed;
var int StatsRoundServer_ThawedOther;
var int StatsRoundServer_Kills;


// Server side data, to be copied to the data that is replicated to the client
var int StatsServerP4R_Shots[`STATS_Types];
var int StatsServerP4R_HitsTo[`STATS_Types];
var int StatsServerP4R_HitsBy[`STATS_Types];
var int StatsServerP4R_DamageTo[`STATS_Types];
var int StatsServerP4R_DamageBy[`STATS_Types];
var int StatsServerP4R_KilledTo[`STATS_Types];
var int StatsServerP4R_KilledBy[`STATS_Types];
var int StatsServerP4R_DirtyCTR;

// Server side data that gets replicated to the client
var enum ESpecialPlayerState
{
	SPS_Alive,
	SPS_Frozen,
	SPS_VeryDead
} SpecialPlayerState;
var UTPawn MyPawn; // To get to the Pawn if we have no access to the controller
var bool bIgnoreMe; // E.g. downloading a map
var vector PawnLocation;
var int TotalHealth;
var int Stats_Kills;
var int Stats_Thawed;
var int Stats_ThawedOther;
var int Stats_Shots[`STATS_Types];
var int Stats_HitsTo[`STATS_Types];
var int Stats_HitsBy[`STATS_Types];
var int Stats_DamageTo[`STATS_Types];
var int Stats_DamageBy[`STATS_Types];
var int Stats_KilledTo[`STATS_Types];
var int Stats_KilledBy[`STATS_Types];
var int Stats_DirtyCTR;

// Client side data
var() ESpecialPlayerState ClientSpecialPlayerState;
var Material StandardSkin;
var bool bShowStats;
var BattlePRI ShowBPRI;
var int StatsPres_Shots[`STATS_Types];
var int StatsPres_HitsTo[`STATS_Types];
var int StatsPres_HitsBy[`STATS_Types];
var int StatsPres_DamageTo[`STATS_Types];
var int StatsPres_DamageBy[`STATS_Types];
var int StatsPres_KilledTo[`STATS_Types];
var int StatsPres_KilledBy[`STATS_Types];
var int StatsPres_Present[`STATS_Types];
var int StatsPres_DirtyCTR;
var int StatsPresTotal_Shots;
var int StatsPresTotal_HitsTo;
var int StatsPresTotal_HitsBy;
var int StatsPresTotal_DamageTo;
var int StatsPresTotal_DamageBy;
var int StatsPresTotal_KilledTo;
var int StatsPresTotal_KilledBy;

replication
{
  if (Role == ROLE_Authority)
    SpecialPlayerState, MyPawn, PawnLocation, TotalHealth, Stats_Shots, Stats_HitsTo, Stats_HitsBy, 
    Stats_DamageTo, Stats_DamageBy, Stats_KilledTo, Stats_KilledBy, Stats_DirtyCTR, Stats_Kills,
    Stats_Thawed, Stats_ThawedOther, bIgnoreMe;
}

function ResetRoundStats ()
{
  local int i;

  for (i=0; i<`STATS_Types; i++) {
    StatsRoundServer_Shots[i] = 0;
    StatsRoundServer_HitsTo[i] = 0;
    StatsRoundServer_HitsBy[i] = 0;
    StatsRoundServer_DamageTo[i] = 0;
    StatsRoundServer_DamageBy[i] = 0;
    StatsRoundServer_KilledTo[i] = 0;
    StatsRoundServer_KilledBy[i] = 0;
  }
  StatsRoundServer_Thawed = 0;
  StatsRoundServer_ThawedOther = 0;
  StatsRoundServer_Kills = 0;
}

simulated function Stats_Prepare ()
{
  local int i;

  if (StatsPres_DirtyCTR != Stats_DirtyCTR) {
    StatsPres_DirtyCTR = Stats_DirtyCTR;
    StatsPresTotal_Shots = 0;
    StatsPresTotal_HitsTo = 0;
    StatsPresTotal_HitsBy = 0;
    StatsPresTotal_DamageTo = 0;
    StatsPresTotal_DamageBy = 0;
    StatsPresTotal_KilledTo = 0;
    StatsPresTotal_KilledBy = 0;
    for (i=0; i<`STATS_Types; i++) {
      StatsPres_Shots[i] = Stats_Shots[i];
      StatsPres_HitsTo[i] = Stats_HitsTo[i];
      StatsPres_HitsBy[i] = Stats_HitsBy[i];
      StatsPres_DamageTo[i] = Stats_DamageTo[i];
      StatsPres_DamageBy[i] = Stats_DamageBy[i];
      StatsPres_KilledTo[i] = Stats_KilledTo[i];
      StatsPres_KilledBy[i] = Stats_KilledBy[i];
      StatsPres_Present[i] = StatsPres_Shots[i] + StatsPres_HitsTo[i] + StatsPres_HitsBy[i] + StatsPres_DamageTo[i] 
                             + StatsPres_DamageBy[i] + StatsPres_KilledTo[i] + StatsPres_KilledBy[i];
    }
    StatsPres_Shots[`STATS_DMG_ShockPrimary]     -= StatsPres_Shots[`STATS_DMG_ShockCombo];
    StatsPres_Shots[`STATS_DMG_ShockBall]        -= StatsPres_Shots[`STATS_DMG_ShockCombo];
    for (i=0; i<`STATS_Types; i++) {
      StatsPresTotal_Shots += StatsPres_Shots[i];
      StatsPresTotal_HitsTo += StatsPres_HitsTo[i];
      StatsPresTotal_HitsBy += StatsPres_HitsBy[i];
      StatsPresTotal_DamageTo += StatsPres_DamageTo[i];
      StatsPresTotal_DamageBy += StatsPres_DamageBy[i];
      StatsPresTotal_KilledTo += StatsPres_KilledTo[i];
      StatsPresTotal_KilledBy += StatsPres_KilledBy[i];
    }
    StatsPres_DamageTo[`STATS_DMG_SniperPrimary] += StatsPres_DamageTo[`STATS_DMG_SniperHeadShot];
    StatsPres_DamageBy[`STATS_DMG_SniperPrimary] += StatsPres_DamageBy[`STATS_DMG_SniperHeadShot];
    StatsPres_KilledTo[`STATS_DMG_SniperPrimary] += StatsPres_KilledTo[`STATS_DMG_SniperHeadShot];
    StatsPres_KilledBy[`STATS_DMG_SniperPrimary] += StatsPres_KilledBy[`STATS_DMG_SniperHeadShot];
  }
}

simulated function string Stats_DamageIndex2Name (int index)
{
  if (index == `STATS_DMG_ImpactHammer) {
    return "Impact Hammer";
  } else if (index == `STATS_DMG_Enforcer) {
    return "Enforcer";
  } else if (index == `STATS_DMG_Bio) {
    return "Bio";
  } else if (index == `STATS_DMG_ShockPrimary) {
    return "Shock Beam";
  } else if (index == `STATS_DMG_ShockBall) {
    return "Shock Ball";
  } else if (index == `STATS_DMG_ShockCombo) {
    return "Shock Combo";
  } else if (index == `STATS_DMG_LinkPlasma) {
    return "Link Plasma";
  } else if (index == `STATS_DMG_LinkBeam) {
    return "Link Beam";
  } else if (index == `STATS_DMG_StingerBullet) {
    return "Stinger Bullet";
  } else if (index == `STATS_DMG_StingerShard) {
    return "Stinger Shard";
  } else if (index == `STATS_DMG_FlakShard) {
    return "Flak Shard";
  } else if (index == `STATS_DMG_FlakShell) {
    return "Flak Shell";
  } else if (index == `STATS_DMG_Rocket) {
    return "Rocket";
  } else if (index == `STATS_DMG_Grenade) {
    return "Grenade";
  } else if (index == `STATS_DMG_SniperPrimary) {
    return "Sniper";
  } else if (index == `STATS_DMG_SniperHeadShot) {
    return "Headshot";
  } else {
    return "???";
  }
}

function int Stats_DamageType2Index (class<DamageType> DamageType)
{
  if (DamageType == class'UTDmgType_ImpactHammer') {
    return `STATS_DMG_ImpactHammer;
  } else if (DamageType == class'UTDmgType_Enforcer' || DamageType == class'UTDmgType_DualEnforcer') {
    return `STATS_DMG_Enforcer;
  } else if (DamageType == class'UTDmgType_BioGoo' || DamageType == class'UTDmgType_BioGooGib' || DamageType == class'UTDmgType_BioGoo_Charged') {
    return `STATS_DMG_Bio;
  } else if (DamageType == class'UTDmgType_ShockPrimary') {
    return `STATS_DMG_ShockPrimary;
  } else if (DamageType == class'UTDmgType_ShockBall') {
    return `STATS_DMG_ShockBall;
  } else if (DamageType == class'UTDmgType_ShockCombo') {
    return `STATS_DMG_ShockCombo;
  } else if (DamageType == class'UTDmgType_LinkPlasma') {
    return `STATS_DMG_LinkPlasma;
  } else if (DamageType == class'UTDmgType_LinkBeam') {
    return `STATS_DMG_LinkBeam;
  } else if (DamageType == class'UTDmgType_StingerBullet') {
    return `STATS_DMG_StingerBullet;
  } else if (DamageType == class'UTDmgType_StingerShard') {
    return `STATS_DMG_StingerShard;
  } else if (DamageType == class'UTDmgType_FlakShard') {
    return `STATS_DMG_FlakShard;
  } else if (DamageType == class'UTDmgType_FlakShell') {
    return `STATS_DMG_FlakShell;
  } else if (DamageType == class'UTDmgType_Rocket' || DamageType == class'UTDmgType_SeekingRocket') {
    return `STATS_DMG_Rocket;
  } else if (DamageType == class'UTDmgType_Grenade') {
    return `STATS_DMG_Grenade;
  } else if (DamageType == class'UTDmgType_SniperPrimary') {
    return `STATS_DMG_SniperPrimary;
  } else if (DamageType == class'UTDmgType_SniperHeadShot') {
    return `STATS_DMG_SniperHeadShot;
  } else {
    return -1;
  }
}

function Stats_RecordShot (int index)
{
  if (!WorldInfo.Game.IsInState('MatchInProgress')) return;  
  StatsServer_Shots[index]++;
  StatsRoundServer_Shots[index]++;
  StatsServer_DirtyCTR++;
}

function Stats_RecordHit (BattlePRI Victim, class<DamageType> DamageType, int Damage)
{
  local int index;

  if (!WorldInfo.Game.IsInState('MatchInProgress')) return;  
  index = Stats_DamageType2Index (DamageType);
  if (index != -1) {
    StatsServer_HitsTo[index]++;
    StatsRoundServer_HitsTo[index]++;
    StatsServer_DamageTo[index] += Damage;
    StatsRoundServer_DamageTo[index] += Damage;
    if (Victim != None) {
      Victim.StatsServer_HitsBy[index]++;
      Victim.StatsRoundServer_HitsBy[index]++;
      Victim.StatsServer_DamageBy[index] += Damage;
      Victim.StatsRoundServer_DamageBy[index] += Damage;
    }
  }
  StatsServer_DirtyCTR++;
}

function Stats_RecordKill (BattlePRI Victim, class<DamageType> DamageType)
{
  local int index;

  if (!WorldInfo.Game.IsInState('MatchInProgress')) return;  
  index = Stats_DamageType2Index (DamageType);
  if (index != -1) {
    if (Victim != Self) StatsServer_KilledTo[index]++; // Count no kill for killing yourself
    if (Victim != Self) StatsRoundServer_KilledTo[index]++; // Count no kill for killing yourself
    if (Victim != None) {
      Victim.StatsServer_KilledBy[index]++;
      Victim.StatsRoundServer_KilledBy[index]++;
    }
  }
  StatsServer_DirtyCTR++;
  if (Victim != Self) Stats_Kills++;
  if (Victim != Self) StatsRoundServer_Kills++;
}

function RequestStatReplication()
{
  local int i;

  if (StatsServer_DirtyCTR != StatsServerP4R_DirtyCTR) {
    StatsServerP4R_DirtyCTR = StatsServer_DirtyCTR;
    for (i=0; i<`STATS_Types; i++) {
      StatsServerP4R_Shots[i] = StatsServer_Shots[i];
      StatsServerP4R_HitsTo[i] = StatsServer_HitsTo[i];
      StatsServerP4R_HitsBy[i] = StatsServer_HitsBy[i];
      StatsServerP4R_DamageTo[i] = StatsServer_DamageTo[i];
      StatsServerP4R_DamageBy[i] = StatsServer_DamageBy[i];
      StatsServerP4R_KilledTo[i] = StatsServer_KilledTo[i];
      StatsServerP4R_KilledBy[i] = StatsServer_KilledBy[i];
    }
  }
}

simulated function Tick (float DeltaTime)
{
  local int i;
 
   if (Role == ROLE_Authority) {
    if ((Controller(Owner) != None) && (UTPawn(Controller(Owner).Pawn) != None) && MyPawn != UTPawn(Controller(Owner).Pawn)) {  
      MyPawn = UTPawn(Controller(Owner).Pawn);
    }
    if (MyPawn != None) {
      if (--TickCtr_Fast<=0) { 
        if (PawnLocation != MyPawn.Location) {
          PawnLocation = MyPawn.Location; 
        } 
        if (TotalHealth != MyPawn.Health + MyPawn.ShieldBeltArmor + MyPawn.VestArmor + MyPawn.ThighpadArmor + MyPawn.HelmetArmor) {
          TotalHealth = MyPawn.Health + MyPawn.ShieldBeltArmor + MyPawn.VestArmor + MyPawn.ThighpadArmor + MyPawn.HelmetArmor;
        }
        TickCtr_Fast = 5 + Rand(2); // Save replication (network) overhead
      }
      if (--TickCtr_Slow<=0) { 
        if (Stats_DirtyCTR != StatsServerP4R_DirtyCTR) {
          Stats_DirtyCTR = StatsServerP4R_DirtyCTR;
          for (i=0; i<`STATS_Types; i++) {
            if (Stats_Shots[i] != StatsServerP4R_Shots[i]) Stats_Shots[i] = StatsServerP4R_Shots[i];
            if (Stats_HitsTo[i] != StatsServerP4R_HitsTo[i]) Stats_HitsTo[i] = StatsServerP4R_HitsTo[i];
            if (Stats_HitsBy[i] != StatsServerP4R_HitsBy[i]) Stats_HitsBy[i] = StatsServerP4R_HitsBy[i];
            if (Stats_DamageTo[i] != StatsServerP4R_DamageTo[i]) Stats_DamageTo[i] = StatsServerP4R_DamageTo[i];
            if (Stats_DamageBy[i] != StatsServerP4R_DamageBy[i]) Stats_DamageBy[i] = StatsServerP4R_DamageBy[i];
            if (Stats_KilledTo[i] != StatsServerP4R_KilledTo[i]) Stats_KilledTo[i] = StatsServerP4R_KilledTo[i];
            if (Stats_KilledBy[i] != StatsServerP4R_KilledBy[i]) Stats_KilledBy[i] = StatsServerP4R_KilledBy[i];
          }
        }
        TickCtr_Slow = 20 + Rand(2); // Do this once every 20 ticks to save replication (network) overhead
      }
    }
  }
  if (MyPawn != None) {
    if (SpecialPlayerState == SPS_Frozen) {
      if (MyPawn.Mesh != None) {
        MyPawn.Mesh.bPauseAnims = true;
      }
    } else {
      MyPawn.SetSkin (None);
      if (MyPawn.Mesh != None) {
        if (MyPawn.Mesh != None) MyPawn.Mesh.bPauseAnims = false;
      }
    }
  }
  if (MyPawn != None && (WorldInfo.NetMode == NM_ListenServer || WorldInfo.NetMode == NM_Client || WorldInfo.NetMode == NM_StandAlone)) {
    i = 0;
    if (WorldInfo != None && BattleGRI(WorldInfo.GRI) != None && BattleGRI(WorldInfo.GRI).LocalBattlePlayerController != None) {
      i = BattleGRI(WorldInfo.GRI).LocalBattlePlayerController.SkinSetting;
    }
    if (SpecialPlayerState == SPS_Frozen) {
      if (i==0) {
        MyPawn.SetSkin (Material'BattleTeamArenaContent.Materials.MT_LiquidBlue');
      } else if (i== 1 || i==4 || i==5 || i==6) {
        MyPawn.SetSkin (Material'BattleTeamArenaContent.Materials.MT_LiquidWhite');
      } else if (i==2 || i==7) {  
        MyPawn.SetSkin (Material'BattleTeamArenaContent.Materials.MT_LiquidGold');
      } else if (i==3) {  
        MyPawn.SetSkin (Material'BattleTeamArenaContent.Materials.MT_Tar');
      }
    } else {
      if (i==4) {
        if (Team.Teamindex==0) {
           BattlePawn(MyPawn).SetPawnSkin (Material'BattleTeamArenaContent.Materials.MT_Red');
        } else {
           BattlePawn(MyPawn).SetPawnSkin (Material'BattleTeamArenaContent.Materials.MT_Blue'); 
        }
      } else if (i==5 || i==7) {
        if (Team.Teamindex==0) {
           BattlePawn(MyPawn).SetPawnSkin (Material'BattleTeamArenaContent.Materials.MT_LiquidRed');
        } else {
           BattlePawn(MyPawn).SetPawnSkin (Material'BattleTeamArenaContent.Materials.MT_LiquidBlue'); 
        }
      } else if (i==6) {
        if (Team.Teamindex==0) {
           BattlePawn(MyPawn).SetPawnSkin (Material'BattleTeamArenaContent.Materials.MT_GlowingRed');
        } else {
           BattlePawn(MyPawn).SetPawnSkin (Material'BattleTeamArenaContent.Materials.MT_GlowingBlue'); 
        }
      } else {
        MyPawn.SetSkin (None);
      }
	}      
  }
  if (MyPawn != None && ClientSpecialPlayerState != SpecialPlayerState) { // Since repnotify does not always seem to work
    ClientSpecialPlayerState = SpecialPlayerState;
    if (SpecialPlayerState == SPS_Frozen) {
      MyPawn.StopFiring ();
    }    
  }
  Super.Tick (DeltaTime);
}

defaultproperties
{
  SpecialPlayerState=SPS_Alive
  Name="BattlePRI"
}