/* 
    BattleTeamArena Copyright (C) 2007-2008 Nico de Vries.

    This file is part of BattleTeamArena.

    BattleTeamArena is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see http://www.gnu.org/licenses/.

    This file also contains code that is derived from code which is
    Copyright 1998-2007 Epic Games, Inc. All Rights Reserved. That part 
    of the code is covered by the Epic license and not by the GPL.
*/  
 
`include (BattleTeamArena/Classes/BattleTeamArena.uci)  

class BattleTeamArena extends UTTeamGame config(BattleTeamArena);

Var array< class<Inventory> > BonusInventory;
var int PreviouslyAliveOnT0, PreviouslyAliveOnT1;

var string RecordedPlayers;

function RecordPlayer (string PlayerName) {
  if (!CheckPlayerReentry (PlayerName)) {
    RecordedPlayers = RecordedPlayers $ ";" $ PlayerName $ ";";
  }
}

function bool CheckPlayerReentry (string PlayerName) {
  return InStr (RecordedPlayers, ";" $ PlayerName $ ";") != -1;
}

function ResetPlayerStorage ()
{
  RecordedPlayers = "";
}

event GetSeamlessTravelActorList(bool bToEntry, out array<Actor> ActorList)
{
  ResetPlayerStorage ();
  Super.GetSeamlessTravelActorList(bToEntry, ActorList);
}

event PostLogin (PlayerController NewPlayer)
{
  Super.PostLogin (NewPlayer);
  if (WorldInfo.GRI.ElapsedTime > 10  && CheckPlayerReentry (NewPlayer.PlayerReplicationInfo.PlayerName)) {
    BattlePRI (NewPlayer.PlayerReplicationInfo).bDoomed = true;
  }
  RecordPlayer (NewPlayer.PlayerReplicationInfo.PlayerName);
}

function ServerBroadcastMessage (int Switch, optional PlayerReplicationInfo RelatedPRI_1, optional PlayerReplicationInfo RelatedPRI_2)
{
  local Controller C;

  foreach WorldInfo.AllControllers(class'Controller', C) {
    if (BattlePlayerController (C) != None) {
      BattlePlayerController (C).ClientShowMessage (Switch, RelatedPRI_1, RelatedPRI_2);
    }
  }
}

function ReduceDamage( out int Damage, pawn injured, Controller instigatedBy, vector HitLocation, out vector Momentum, class<DamageType> DamageType )
{
  if (instigatedBy != None && instigatedBy != injured.Controller && Injured.GetTeamNum() == instigatedBy.GetTeamNum()) {
    if (Injured.Owner != None && Controller(Injured.Owner) != None && Controller(Injured.Owner).PlayerReplicationInfo != None
        && BattlePRI (Controller(Injured.Owner).PlayerReplicationInfo) != None) {
      if (BattlePRI (Controller(Injured.Owner).PlayerReplicationInfo).SpecialPlayerState != SPS_Frozen) {
        InstigatedBy.Pawn.TakeDamage(Damage/2, None, Injured.Location, vect(0,0,0), class'BattleDamageType_ReverseFriendlyFire');
      }
    }
  }
  Super.ReduceDamage (Damage, injured, instigatedBy, HitLocation, Momentum, DamageType);
  if (instigatedBy != None && Injured.Owner != None && Controller(Injured.Owner) != None && Controller(Injured.Owner).PlayerReplicationInfo != None
      && BattlePRI (Controller(Injured.Owner).PlayerReplicationInfo) != None) {
    if (BattlePRI (Controller(Injured.Owner).PlayerReplicationInfo).SpecialPlayerState == SPS_Frozen) {
      if (Injured.GetTeamNum() == instigatedBy.GetTeamNum()) {
        Momentum = Momentum / 0.3; // default TeammateBoost
      }
    }
  }
}

function RestartPlayer(Controller NewPlayer)
{
  if (BattlePRI (NewPlayer.PlayerReplicationInfo).SpecialPlayerState != SPS_VeryDead) {
    Super.RestartPlayer(NewPlayer);
  }
}

function PostBeginPlay()
{
  Super.PostBeginPlay();
  RemoveStuff ();

  if (Role == ROLE_Authority) {
    SetTimer (1.0, True, 'ServerTimer');
  }
}

function RemoveStuff ()
{
  local UTGameObjective GO;
  local UTPickupFactory PU;
  local UTVehicleFactory VF;
  local UTVehicle V;

  // Remove weapons, health, ammo, etc.
  if (BattleGRI(WorldInfo.GRI).RemoveSetting != 1 && BattleGRI(WorldInfo.GRI).RemoveSetting != 3 && BattleGRI(WorldInfo.GRI).RemoveSetting != 4) {
    foreach AllActors(class'UTPickupFactory', PU) { 
      PU.DisablePickup();
    }
  }

  // Remove vehicles, turrets, etc.
  if (BattleGRI(WorldInfo.GRI).RemoveSetting != 2 && BattleGRI(WorldInfo.GRI).RemoveSetting != 3 && BattleGRI(WorldInfo.GRI).RemoveSetting != 4) {
    foreach AllActors(class'UTVehicleFactory', VF) { 
      VF.Destroy();
    }
    foreach AllActors(class'UTVehicle', V) { 
      V.Destroy();
    }
  }

  // Remove objectives
  if (BattleGRI(WorldInfo.GRI).RemoveSetting != 4) {
    foreach AllActors(class'UTGameObjective', GO) {
	  GO.bIsActive = false;
  	  GO.bIsDisabled = true;
	  if (UTOnslaughtPowernode(GO) != none) { // Remove powernode
        GO.DefenderTeamIndex = 255;
	    UTOnslaughtPowernode(GO).bStandalone = false;
	    UTOnslaughtPowernode(GO).GotoState('DisabledNode');
	  } else if (UTOnslaughtPowerCore(GO) != none) { // Remove powercore
  	    GO.DefenderTeamIndex = 255;
	    UTOnslaughtPowerCore(GO).bStandalone = false;
	    UTOnslaughtPowerCore(GO).GotoState('DisabledNode');
	  } else if (UTOnslaughtSpecialObjective(GO) != none) { // Remove other things
        if (UTWarfareBarricade(GO) != none) {
	      UTWarfareBarricade(GO).SetDisabled(true);
	    } else {
          GO.DefenderTeamIndex = 255;
		  UTOnslaughtSpecialObjective(GO).GotoState('DisabledNode');
	    }
	  } else if (UTOnslaughtNodeEnhancement(GO) != none) { // Remove orbs
        if (UTOnslaughtFlagBase(GO) != none) {
          UTOnslaughtFlagBase(GO).DisableOrbs();
	    }
	    GO.DefenderTeamIndex = 255;
	    UTOnslaughtNodeEnhancement(GO).GotoState('DisabledNode');
	  } else if (UTCTFBase(GO) != none)	{ // Remove CTF bases
        GO.DefenderTeamIndex = 255;
	  }
    }
  }
}

function ServerTimer ()
{
  local UTVehicle V;
  
  if (!WorldInfo.Game.IsInState('MatchInProgress')) return; 
  DetectRoundOver (None); // E.g. if a team lost because a player left the server
  if (BattleGRI(WorldInfo.GRI).RemoveSetting != 2 && BattleGRI(WorldInfo.GRI).RemoveSetting != 3 && BattleGRI(WorldInfo.GRI).RemoveSetting != 4) {
    foreach AllActors(class'UTVehicle', V) { // Just in case some vehicle pops up anyway
      V.Destroy();
    }
  }
}

function SetPlayerDefaults(Pawn PlayerPawn)
{
  local int MyTeamScore, OtherTeamScore, ScoreDifference;
  local Controller C;
  local UTWeapon UTW;
  local int i, AmmoToAdd;

  MyTeamScore = PlayerPawn.PlayerReplicationInfo.Team.Score;
  
  // There must be a better way to do this
  foreach WorldInfo.AllControllers(class'Controller', C) { // Also check !C.PlayerReplicationInfo.bOutOfLives ???
    if (C.bIsPlayer && C.GetTeamNum() == 1-PlayerPawn.PlayerReplicationInfo.Team.TeamIndex) {
      OtherTeamScore = C.PlayerReplicationInfo.Team.Score;
    }
  }
  
  ScoreDifference = Min (9, Max (-9, MyTeamScore-OtherTeamScore));
  Super.SetPlayerDefaults (PlayerPawn);
  if (UTPawn (PlayerPawn) != None) {
    if (BattleGRI (WorldInfo.GRI).ChallengeModeSetting == 0) { // Losing extra, winning less
    } else if (BattleGRI (WorldInfo.GRI).ChallengeModeSetting == 1) { // Losing extra
      if (ScoreDifference > 0) ScoreDifference = 0;
    } else { // Nothing
      ScoreDifference = 0;
    }
    UTPawn(PlayerPawn).Health = 100 - 5 * ScoreDifference;
    UTPawn(PlayerPawn).ShieldBeltArmor = 25 - 2*ScoreDifference;
    UTPawn(PlayerPawn).VestArmor = 25 - ScoreDifference;
    UTPawn(PlayerPawn).ThighpadArmor = 25 - ScoreDifference;
    UTPawn(PlayerPawn).HelmetArmor = 25 - ScoreDifference;
  }  

  for (i=0; i<BattleGRI (WorldInfo.GRI).WeaponsSetting-1; i++) {
    if (UTPawn(PlayerPawn).FindInventoryType(BonusInventory[i]) == None) {
      UTPawn(PlayerPawn).CreateInventory(BonusInventory[i], false); 
    }
  } 

  ForEach UTPawn(PlayerPawn).InvManager.InventoryActors (class'UTWeapon', UTW) {
    if (!UTW.bSuperWeapon) {
      AmmoToAdd = int (UTW.default.AmmoCount * (BattleGRI (WorldInfo.GRI).AmmoSetting-100.0)/100.0);
      if (UTW.default.AmmoCount + AmmoToAdd == 0) AmmoToAdd = 1 - UTW.default.AmmoCount; // At least 1 shot
      UTW.AddAmmo (AmmoToAdd); // Automatically reduced by UT3
    }
  }
}

// This is mainly ment to detect players downloading a map, might need revisiting
function bool IgnorePlayer (Controller CT)
{
  local Controller C;
  local bool Found1RealPlayer;  

  Found1RealPlayer = false;
  foreach WorldInfo.AllControllers(class'Controller', C) {
    if (PlayerController (C) != None && LLIgnorePlayer (C)) {
      Found1RealPlayer = true;
    }
  }
  if (!Found1RealPlayer) return false; // Assume something is wrong with the detection, 0 players is not useful anyway
  return LLIgnorePlayer (CT);
}

function bool LLIgnorePlayer (Controller C)
{
  local bool Ret;

  Ret = false;
  if (C == None) {
    Ret = true;
  } else if (C.PlayerReplicationInfo == None) {
    Ret = true;
  } else if (PlayerController(C) != None && !PlayerController(C).CanRestartPlayer()) { // Checks bOnlySpectator and HasClientLoadedCurrentWorld
    Ret = true;
  } else if (BattlePRI (C.PlayerReplicationInfo).MyPawn == None) {
    Ret = true;
  } else if (BattlePRI (C.PlayerReplicationInfo).MyPawn.Location == Vect (0,0,0)) {
    Ret = true;
  }
  if (BattlePRI (C.PlayerReplicationInfo) != None && BattlePRI (C.PlayerReplicationInfo).SpecialPlayerState != SPS_Alive) Ret = false; // Only ignore living players
  if (PlayerController(C) == None) Ret = false; // Only ignore players (never ignore a bot)
  BattlePRI (C.PlayerReplicationInfo).bIgnoreMe = Ret;
  return Ret;
}

function DetectRoundOver (PlayerReplicationInfo PRI) // PRI of killer
{
  local Controller C;
  local bool AliveT0, AliveT1;
  local int AliveOnT0, AliveOnT1;
  local bool SomeoneOnT0, SomeoneOnT1;
  local int WinningTeam;

  if (WorldInfo == None || WorldInfo.Game == None) return;

  if (!WorldInfo.Game.IsInState('MatchInProgress')) return; // Only one team can win a around, only once

  // Detect if one team is completely frozen
  AliveT0 = false;
  AliveT1 = false;
  foreach WorldInfo.AllControllers(class'Controller', C) {
    if (C != none && C.PlayerReplicationInfo != none) {
      RecordPlayer (C.PlayerReplicationInfo.PlayerName);
    }
    if (C != none && C.PlayerReplicationInfo != none && !C.PlayerReplicationInfo.bOnlySpectator && !IgnorePlayer (C)) {
      if (C.GetTeamNum() == 0 && BattlePRI (C.PlayerReplicationInfo).SpecialPlayerState == SPS_Alive) {
        AliveT0 = true;
        AliveOnT0++;
      }
      if (C.GetTeamNum() == 1 && BattlePRI (C.PlayerReplicationInfo).SpecialPlayerState == SPS_Alive) {
        AliveT1 = true;
        AliveOnT1++;
      }
      if (C.GetTeamNum() == 0) SomeoneOnT0 = true;
      if (C.GetTeamNum() == 1) SomeoneOnT1 = true;
    }
  }
  if (!SomeoneOnT0 || !SomeoneOnT1) { // One of the teams is empty
    if (BattleGRI (WorldInfo.GRI).RemainingTime > BattleGRI (WorldInfo.GRI).RemainingTimeRoundOffset) { // Round time left
      return;
    }
  }
  if (AliveT0 && AliveT1) { // Last one standing
    if (AliveOnT0 == 1 && PreviouslyaliveOnT0 != 1) {
      foreach WorldInfo.AllControllers(class'Controller', C) {
        if (C != none && C.PlayerReplicationInfo != none && !C.PlayerReplicationInfo.bOnlySpectator && !IgnorePlayer (C)) {
          if (C.GetTeamNum() == 0 && BattlePRI (C.PlayerReplicationInfo).SpecialPlayerState == SPS_Alive) {
            if (BattlePlayerController (C) != None) BattlePlayerController (C).ServerShowMessage (7);
          }
        }
      }
    }
    if (AliveOnT1 == 1 && PreviouslyaliveOnT1 != 1) {
      foreach WorldInfo.AllControllers(class'Controller', C) {
        if (C != none && C.PlayerReplicationInfo != none && !C.PlayerReplicationInfo.bOnlySpectator && !IgnorePlayer (C)) {
          if (C.GetTeamNum() == 1 && BattlePRI (C.PlayerReplicationInfo).SpecialPlayerState == SPS_Alive) {
            if (BattlePlayerController (C) != None) BattlePlayerController (C).ServerShowMessage (7);
          }
        }
      }
    }
  }
  PreviouslyaliveOnT0 = AliveOnT0;
  PreviouslyaliveOnT1 = AliveOnT1;
  if (!AliveT0 || !AliveT1) {
    if (AliveT0) WinningTeam = 0; else WinningTeam = 1;

    // If the killer is on the losing team (e.g. suicide) replace with someone from the winning team
    if (PRI != None && PRI.Team.TeamIndex != WinningTeam) PRI = None;

    // If the killer is unknown (or not on the winning team) locate the player with the highest score instead
    if (PRI == None) { 
      foreach WorldInfo.AllControllers(class'Controller', C) { // Also check !C.PlayerReplicationInfo.bOutOfLives ???
        if (C.bIsPlayer && C.GetTeamNum() == WinningTeam
            && ((PRI == None) || (C.PlayerReplicationInfo != None && C.PlayerReplicationInfo.Score >= PRI.Score))) {
          PRI = c.PlayerReplicationInfo;
        }
      }
    }

    if (PRI != None) {
      // Increase teams score of winning team
      PRI.Team.Score += 1.0;
      PRI.Team.bForceNetUpdate = true;

      // End of game or next round
      if (!CheckScore(PRI)) { // Goal reached or out of time
        // Next round
        AnnounceScore (PRI.Team.TeamIndex);
        BattleGRI (WorldInfo.GRI).bRoundIsOver = true; // new round
        CountDown = 8;
        GoToState ('PendingMatch');
        CalculateRoundStatistics();
      }
    }
  }
}

function CalculateRoundStatistics ()
{
  local Controller C;
  local int i;
  local int Kills, Kills_Total, Kills_Max;
  local int Damage, Damage_Total, Damage_Max;
  local int Headshots, Headshots_Total, Headshots_Max;
  local int Thaws, Thaws_Total, Thaws_Max;
  local BattlePRI Kills_MaxBPRI, Damage_MaxBPRI, Headshots_MaxBPRI, Thaws_MaxBPRI, BPRI;

  foreach WorldInfo.AllControllers(class'Controller', C) {
    BPRI = BattlePRI (C.PlayerReplicationInfo);
    if (BPRI != None) {
      Damage = 0;      
      for (i=0; i<`STATS_Types; i++) {
        Damage += BPRI.StatsRoundServer_DamageTo[i];
      }
      Damage_Total += Damage;
      if (Damage_Max <= Damage) {
        Damage_MaxBPRI = BPRI;
        Damage_Max = Damage;
      }  

      Thaws = BPRI.StatsRoundServer_ThawedOther;
      Thaws_Total += Thaws;
      if (Thaws_Max <= Thaws) {
        Thaws_MaxBPRI = BPRI;
        Thaws_Max = Thaws;
      }  

      HeadShots = BPRI.StatsRoundServer_HitsTo[`STATS_DMG_SniperHeadShot];
      HeadShots_Total += HeadShots;
      if (HeadShots_Max <= HeadShots) {
        HeadShots_MaxBPRI = BPRI;
        HeadShots_Max = HeadShots;
      }  

      Kills = BPRI.StatsRoundServer_Kills;
      Kills_Total += Kills;
      if (Kills_Max <= Kills) {
        Kills_MaxBPRI = BPRI;
        Kills_Max = Kills;
      }  
    }
  }
  BattleGRI (WorldInfo.GRI).LastRound_Damage = Damage_Total;
  BattleGRI (WorldInfo.GRI).LastRound_MaxDamage = Damage_Max;
  if (Kills_MaxBPRI != None) {
    BattleGRI (WorldInfo.GRI).LastRound_MaxDamageBy = Damage_MaxBPRI.PlayerName;
  } else {
    BattleGRI (WorldInfo.GRI).LastRound_MaxDamageBy = "";
  }
  BattleGRI (WorldInfo.GRI).LastRound_Thaws = Thaws_Total;
  BattleGRI (WorldInfo.GRI).LastRound_MaxThaws = Thaws_Max;
  if (Thaws_MaxBPRI != None) {
    BattleGRI (WorldInfo.GRI).LastRound_MaxThawsBy = Thaws_MaxBPRI.PlayerName;
  } else {
    BattleGRI (WorldInfo.GRI).LastRound_MaxThawsBy = "";
  }
  BattleGRI (WorldInfo.GRI).LastRound_HeadShots = HeadShots_Total;
  BattleGRI (WorldInfo.GRI).LastRound_MaxHeadShots = HeadShots_Max;
  if (HeadShots_MaxBPRI != None) {
    BattleGRI (WorldInfo.GRI).LastRound_MaxHeadShotsBy = HeadShots_MaxBPRI.PlayerName;
  } else {
    BattleGRI (WorldInfo.GRI).LastRound_MaxHeadShotsBy = "";
  }
  BattleGRI (WorldInfo.GRI).LastRound_Kills = Kills_Total;
  BattleGRI (WorldInfo.GRI).LastRound_MaxKills = Kills_Max;
  if (Kills_MaxBPRI != None) {
    BattleGRI (WorldInfo.GRI).LastRound_MaxKillsBy = Kills_MaxBPRI.PlayerName;
  } else {
    BattleGRI (WorldInfo.GRI).LastRound_MaxKillsBy = "";
  }
}

function bool CheckScore (PlayerReplicationInfo Scorer)
{
	if (CheckMaxLives(Scorer))
	{
		return false;
	}
	else if (!bOverTime && GoalScore == 0)
	{
		return false;
	}
	else if (Scorer != None && Scorer.Team != None && Scorer.Team.Score >= GoalScore)
	{
		EndGame(Scorer,"teamscorelimit");
		return true;
	}
	else if (Scorer != None && bOverTime)
	{
		EndGame(Scorer,"timelimit");
		return true;
	}
	else
	{
		return false;
	}
}

function StartNewRound ()
{
  local Controller C;

  foreach WorldInfo.AllControllers(class'Controller', C) {
    if (C != none && C.PlayerReplicationInfo != none && !C.PlayerReplicationInfo.bOnlySpectator) {
      if (UTPlayerController(C) != None) {
        UTPlayerController(C).bBehindView = false;
      }
      BattlePRI (C.PlayerReplicationInfo).SpecialPlayerState = SPS_Alive;
      BattlePRI (C.PlayerReplicationInfo).ResetRoundStats(); 
      if (PlayerController(C) != none) PlayerController(C).ResetCameraMode();
      if(C.Pawn != None) C.Pawn.Destroy();
      if (UTBot (C) != None) {
        C.GotoState('Reboot');
      } else {
        WorldInfo.Game.ReStartPlayer(C);
      }
    }
  }
}

function Special_Super_Killed( Controller Killer, Controller KilledPlayer, Pawn KilledPawn, class<DamageType> damageType) // Derived from GameInfo.uc
{
    if( KilledPlayer != None && KilledPlayer.bIsPlayer )
	{
		KilledPlayer.PlayerReplicationInfo.Deaths += 1;
		KilledPlayer.PlayerReplicationInfo.SetNetUpdateTime(FMin(KilledPlayer.PlayerReplicationInfo.NetUpdateTime, WorldInfo.TimeSeconds + 0.3 * FRand()));
		BroadcastDeathMessage(Killer, KilledPlayer, damageType);
	}

    if( KilledPlayer != None )
	{
		ScoreKill(Killer, KilledPlayer);
	}

//	DiscardInventory(KilledPawn, Killer);
    NotifyKilled(Killer, KilledPlayer, KilledPawn);
}

function Special_Killed( Controller Killer, Controller KilledPlayer, Pawn KilledPawn, class<DamageType> damageType ) // Derived from UTGameInfo.uc
{
	local bool		bEnemyKill;
	local UTPlayerReplicationInfo KillerPRI, KilledPRI;
	local UTVehicle V;

	if ( UTBot(KilledPlayer) != None )
		UTBot(KilledPlayer).WasKilledBy(Killer);

	if ( Killer != None )
		KillerPRI = UTPlayerReplicationInfo(Killer.PlayerReplicationInfo);
	if ( KilledPlayer != None )
		KilledPRI = UTPlayerReplicationInfo(KilledPlayer.PlayerReplicationInfo);

	bEnemyKill = ( ((KillerPRI != None) && (KillerPRI != KilledPRI) && (KilledPRI != None)) && (!bTeamGame || (KillerPRI.Team != KilledPRI.Team)) );

	if ( (KillerPRI != None) && UTVehicle(KilledPawn) != None )
	{
		KillerPRI.IncrementVehicleKillStat(UTVehicle(KilledPawn).GetVehicleKillStatName());
	}
	if ( KilledPRI != None )
	{
		KilledPRI.LastKillerPRI = KillerPRI;

		if ( class<UTDamageType>(DamageType) != None )
		{
			class<UTDamageType>(DamageType).static.ScoreKill(KillerPRI, KilledPRI, KilledPawn);
		}
		else
		{
			// assume it's some kind of environmental damage
			if ( KillerPRI == KilledPRI )
			{
				KilledPRI.IncrementSuicideStat('SUICIDES_ENVIRONMENT');
			}
			else
			{
				if ( KillerPRI != None )
					KillerPRI.IncrementKillStat('KILLS_ENVIRONMENT');
				KilledPRI.IncrementDeathStat('DEATHS_ENVIRONMENT');
			}
		}
		if ( KilledPRI.Spree > 4 )
		{
			EndSpree(KillerPRI, KilledPRI);
		}
		else
		{
			KilledPRI.Spree = 0;
		}
		if ( KillerPRI != None )
		{
			KillerPRI.LogMultiKills(bEnemyKill);

			if ( !bFirstBlood && bEnemyKill )
			{
				bFirstBlood = True;
				BroadcastLocalizedMessage( class'UTFirstBloodMessage', 0, KillerPRI );
				KillerPRI.IncrementEventStat('EVENT_FIRSTBLOOD');
			}
			if ( KillerPRI != KilledPRI && (!bTeamGame || (KilledPRI.Team != KillerPRI.Team)) )
			{
				KillerPRI.IncrementSpree();
			}
		}
	}
    Special_Super_Killed(Killer, KilledPlayer, KilledPawn, damageType);

    if ( ((WorldInfo.NetMode == NM_Standalone) || (SinglePlayerMissionID != INDEX_NONE)) && (PlayerController(KilledPlayer) != None) )
    {
		// tell bots not to get into nearby vehicles
		for ( V=VehicleList; V!=None; V=V.NextVehicle )
			if ( WorldInfo.GRI.OnSameTeam(KilledPlayer,V) )
				V.PlayerStartTime = 0;
	}
}

event InitGame(string Options, out string ErrorMessage)
{
  Super.InitGame(Options, ErrorMessage);
  if (TimeLimit==0) {
	TimeLimit = 5999;
	AutomatedPerfRemainingTime = 60 * TimeLimit;
  }
  UTGame(WorldInfo.Game).AddGameRules(class'BattleGameRules');
}

auto State PendingMatch
{
  function EndState(Name NextStateName)
  {
    Super.EndState (NextStateName);
    StartNewRound();
    BattleGRI (WorldInfo.GRI).RemainingTimeRoundOffset = Max (0, BattleGRI (WorldInfo.GRI).RemainingTime - BattleGRI (WorldInfo.GRI).RoundLengthSetting);
    BattleGRI (WorldInfo.GRI).bRoundIsOver = false; // new round
  }

	function Timer()
	{
		local PlayerController P;
		local bool bReady;
		local UTBot B;

		Global.Timer();

		// first check if there are enough net players, and enough time has elapsed to give people
		// a chance to join
		if ( NumPlayers == 0 )
		{
			bWaitForNetPlayers = true;

			if (bWarmupRound)
			{
				WarmupRemaining = WarmupTime;
				GameReplicationInfo.RemainingTime = WarmupRemaining;
			}
		}
		else
		{
			foreach WorldInfo.AllControllers(class'UTBot', B)
			{
				if (TooManyBots(B))
				{
					B.Destroy();
				}
			}

			AddInitialBots();

			if (bWarmupRound)
			{
				if (WarmupRemaining > 0)
				{
					WarmupRemaining--;
					GameReplicationInfo.RemainingTime = WarmupRemaining;
					if (WarmupRemaining % 60 == 0)
					{
						GameReplicationInfo.RemainingMinute = WarmupRemaining;
					}
			   		return;
				}
				else if (WarmupRemaining == 0)
				{
					WarmupRemaining = -1;
					UTGameReplicationInfo(GameReplicationInfo).bWarmupRound = false;
			   		ResetLevel();
				}
			}
		}

		if ( bWaitForNetPlayers && (WorldInfo.NetMode != NM_Standalone) )
		{
			if ( (NumPlayers >= MinNetPlayers) && (NumPlayers > 0) )
				PendingMatchElapsedTime++;
			else
				PendingMatchElapsedTime = 0;
			if ( (NumPlayers == MaxPlayers) || (PendingMatchElapsedTime > NetWait) )
			{
				// wait until players finish clientside processing (or it times out)
				if (PendingMatchElapsedTime <= ClientProcessingTimeout)
				{
					if (!InitialProcessingIsComplete())
					{
						PlayStartupMessage();
						return;
					}
				}
				bWaitForNetPlayers = false;
				CountDown = Default.CountDown;
			}
			else
			{
				PlayStartupMessage();
				return;
			}
		}
		else if (WorldInfo.NetMode == NM_Standalone && !InitialProcessingIsComplete())
		{
			// PlayStartupMessage() intentionally left out here (mesh construction messsage should be visible)
			return;
		}

		// check if players are ready
		bReady = true;

		StartupStage = 1;
		if ( !bStartedCountDown && (bPlayersMustBeReady || (WorldInfo.NetMode == NM_Standalone)) )
		{
			foreach WorldInfo.AllControllers(class'PlayerController', P)
			{
				if ( P.PlayerReplicationInfo != None && P.bIsPlayer && P.PlayerReplicationInfo.bWaitingPlayer
					&& !P.PlayerReplicationInfo.bReadyToPlay )
				{
					bReady = false;
				}
			}
		}
		if ( bReady && SinglePlayerMissionID == INDEX_None )
		{

			if (!bStartedCountDown)
			{
				if (DemoPrefix != "")
				{
					ConsoleCommand("demorec" @ DemoPrefix $ "-%td");
				}
				bStartedCountDown = true;
			}
			CountDown--;
			if ( CountDown <= 0 )
				StartMatch();
			else
				StartupStage = CountDown + 10;
		}
		if (Countdown < 6) PlayStartupMessage(); // 5 4 3 2 1 
	}
}

defaultproperties
{
  PlayerControllerClass=class'BattlePlayerController'
  PlayerReplicationInfoClass=class'BattlePRI'
  GameReplicationInfoClass=Class'BattleGRI'
  DefaultPawnClass=Class'BattlePawn'
  BotClass=Class'BattleBotController'
  HUDType=Class'BattleHUD'
  DefaultEnemyRosterClass="UTGame.UTTeamInfo"
  TeamAIType(0)=Class'BattleTeamAI'
  TeamAIType(1)=Class'BattleTeamAI'
  Acronym="BTA"
  Description="Two teams duke it out in a quest for battlefield supremacy."
  EndOfMatchRulesTemplateStr_Scoring="First team to win `g rounds wins."
  EndOfMatchRulesTemplateStr_Time="Team with most rounds in `t mins. wins."
  TeamScoreMessageClass=Class'UTGameContent.UTTeamScoreMessage'
  CountDown=4
  GoalScore=10
  bAllowNonTeamChat=true
  bForceAllRed=false
  bPlayersBalanceTeams=true
  bAllowHoverboard=false
  bAllowMapVoting=true
  bAllowTranslocator=false
  bScoreTeamKills=false
  MapPrefixes(0)="DM"
  MapPrefixes(1)="CTF"
  MapPrefixes(2)="VCTF"
  MapPrefixes(3)="WAR"
  MapPrefixes(4)="INV"
  MapPrefixes(5)="BTA"
  DefaultInventory(0)=Class'BattleWeapon_Enforcer'
  DefaultInventory(1)=Class'BattleWeapon_ImpactHammer'
  BonusInventory(0)=Class'BattleWeapon_SniperRifle'
  BonusInventory(1)=Class'BattleWeapon_ShockRifle'
  BonusInventory(2)=Class'BattleWeapon_LinkGun'
  BonusInventory(3)=Class'BattleWeapon_RocketLauncher'
  BonusInventory(4)=Class'BattleWeapon_FlakCannon'
  BonusInventory(5)=Class'BattleWeapon_Stinger'
  BonusInventory(6)=Class'BattleWeapon_BioRifle'
  BonusInventory(7)=Class'UTWeap_Avril'
  GameName="Battle Team Arena"
}