/* 
    BattleTeamArena Copyright (C) 2007-2008 Nico de Vries.

    This file is part of BattleTeamArena.

    BattleTeamArena is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see http://www.gnu.org/licenses/.
    
    This file also contains code that is derived from code which is
    Copyright 1998-2007 Epic Games, Inc. All Rights Reserved. That part 
    of the code is covered by the Epic license and not by the GPL.
*/ 

class BattlePawn extends UTPawn;

var Vector CampLocation;
var int CampCounter;
var int FiveCounter;
var int SuperFreezeCounter;
var int SuperFreezeMovesCounter;
var float LastHurtBySlimeOrLava;
var bool bCanBeAutonomous;
var bool bSpecialFallDamage;

simulated event PostBeginPlay()
{
  super.PostBeginPlay();
  if (Role == ROLE_Authority) {
    SetTimer (1.0, True, 'ServerTimer');
  }
}

simulated function SetPawnSkin (Material NewMaterial)
{
	local int i;

	// Replicate the Material to remote clients
	ReplicatedBodyMaterial = NewMaterial;

	if (VerifyBodyMaterialInstance())		// Make sure we have setup the BodyMaterialInstances array
	{
		// Set the skin
		if (NewMaterial == None)
		{
			for (i = 0; i < Mesh.SkeletalMesh.Materials.length; i++)
			{
				Mesh.SetMaterial(i, BodyMaterialInstances[i]);
			}
		}
		else
		{
			for (i = 0; i < Mesh.SkeletalMesh.Materials.length; i++)
			{
				Mesh.SetMaterial(i, NewMaterial);
			}
		}

		SetArmsSkin(NewMaterial);
	}
}

function int RODN (int NumberBeforeRoundOvertime, int NumberAfterRoundOvertime) // Round Overtime Dependent Number
{ 
  if (BattleGRI (WorldInfo.GRI).RemainingTime > BattleGRI (WorldInfo.GRI).RemainingTimeRoundOffset) {
    return NumberBeforeRoundOvertime;
  } else {
    return NumberAfterRoundOvertime;
  }
}

function ServerTimer ()
{
  local Controller C;
  local Controller CC;
  local float Distance;
  local int HealthCtr;
  local int HealthSum;
  local bool Five;

  if (!WorldInfo.Game.IsInState('MatchInProgress')) return;  
  
  if (++FiveCounter == 5) {
    FiveCounter = 0;
    Five = true;
  } else {
    Five = false;
  }

  if (Owner != None && Controller(Owner) != None) {
    C = Controller(Owner);
    if (C.PlayerReplicationInfo != None && BattlePRI (C.PlayerReplicationInfo) != None) {
      if (BattlePRI (C.PlayerReplicationInfo).bDoomed) {
        BattlePRI (C.PlayerReplicationInfo).bDoomed = false;
        TakeDamage(500, None, Location, vect(0,0,0), class'BattleDamageType_LateEntry');
        return;
      }
      if (BattlePRI (C.PlayerReplicationInfo).SpecialPlayerState == SPS_Frozen) {

        // To make sure
        StopFiring();

        // Slowly heal (unfreeze) the player
        if (Health < 100) Health += 2;

        // Unfreeze if team mate is close
        if (Health < 100) {
          foreach WorldInfo.AllControllers(class'Controller', CC) {
            if (CC != C && CC != none && CC.Pawn != None && CC.PlayerReplicationInfo != None && !CC.PlayerReplicationInfo.bOnlySpectator
                && BattlePRI (CC.PlayerReplicationInfo) != None
                && BattlePRI (CC.PlayerReplicationInfo).SpecialPlayerState == SPS_Alive
                && C.GetTeamNum() == CC.GetTeamNum()) {
              Distance = VSize(CC.Pawn.Location - Location);
              if (Distance <= 100) {
                Health += 10;
                if (BattlePlayerController(Owner) != None) {
                  BattlePlayerController(Owner).ServerShowMessage  (2, CC.PlayerReplicationInfo);
                }
                if (BattlePlayerController(CC) != None && Controller(Owner) != None) {
                  BattlePlayerController(CC).ServerShowMessage (8, Controller(Owner).PlayerReplicationInfo);
                }
                HealthSum += CC.Pawn.Health;
                HealthCtr++;
                if (BattlePawn (CC.Pawn).CampCounter > 5) BattlePawn (CC.Pawn).CampCounter = 5; // Unfreezing does not count as camping
              } else if (Distance <= 200) {
                Health += 5;
                if (BattlePlayerController(Owner) != None) {
                  BattlePlayerController(Owner).ServerShowMessage  (3, CC.PlayerReplicationInfo);
                }
                if (BattlePlayerController(CC) != None && Controller(Owner) != None) {
                  BattlePlayerController(CC).ServerShowMessage (9, Controller(Owner).PlayerReplicationInfo);
                }
                HealthSum += CC.Pawn.Health;
                HealthCtr++;
                if (BattlePawn (CC.Pawn).CampCounter > 5) BattlePawn (CC.Pawn).CampCounter = 5; // Unfreezing does not count as camping
              }
            }
          }
        }
        
        // Reward players for bringing me back alive, and unfreeze me
        if (Health >= 100) { 
          foreach WorldInfo.AllControllers(class'Controller', CC) {
            if (CC != C && CC != none && CC.Pawn != None && CC.PlayerReplicationInfo != None && !CC.PlayerReplicationInfo.bOnlySpectator
                && BattlePRI (CC.PlayerReplicationInfo) != None
                && BattlePRI (CC.PlayerReplicationInfo).SpecialPlayerState == SPS_Alive
                && C.GetTeamNum() == CC.GetTeamNum()) {
              CC.PlayerReplicationInfo.Score += 2; // Reward player for bringing me back alive
              BattlePRI (CC.PlayerReplicationInfo).Stats_ThawedOther++;
              BattlePRI (CC.PlayerReplicationInfo).StatsRoundServer_ThawedOther++;
             }
          }
          if (HealthCtr > 0) {
            Unfreeze (BattlePRI (C.PlayerReplicationInfo), HealthSum/HealthCtr);
          } else {
            Unfreeze (BattlePRI (C.PlayerReplicationInfo));
          }
          BattlePRI (C.PlayerReplicationInfo).Stats_Thawed++;
          BattlePRI (C.PlayerReplicationInfo).StatsRoundServer_Thawed++;
          BattleTeamArena(Worldinfo.Game).ServerBroadcastMessage (6, C.PlayerReplicationInfo);
        }
      } else if (BattlePRI (C.PlayerReplicationInfo).SpecialPlayerState == SPS_Alive) {

        // Gain health if a team mate is close (FREEZETAG ONLY)
        if (BattleFreezetagArena (WorldInfo.Game) != None) {
          if (Health < 100 && CampCounter <= 15) { // No healing while camping
            foreach WorldInfo.AllControllers(class'Controller', CC) {
              if (CC != C && CC != none && CC.Pawn != None && CC.PlayerReplicationInfo != None && !CC.PlayerReplicationInfo.bOnlySpectator
                  && BattlePRI (CC.PlayerReplicationInfo) != None
                  && BattlePRI (CC.PlayerReplicationInfo).SpecialPlayerState == SPS_Alive
                  && C.GetTeamNum() == CC.GetTeamNum()) {
                Distance = VSize(CC.Pawn.Location - Location); 
                if (CC.Pawn.Health > Health) {
                  if (Distance <= 100) {
                    Health += 10;
                  } else if (Distance <= 200) {
                    Health += 5;
                  }
                  if (Health >= CC.Pawn.Health) Health = CC.Pawn.Health;
                }
              }
            }
          }
        }

        // Camping detection
        Distance = VSize(C.Pawn.Location - CampLocation); 
        if (Distance < 500 && BattlePRI (C.PlayerReplicationInfo).SpecialPlayerState != SPS_Frozen) {
          CampCounter++;
          if (CampCounter == BattleGRI (WorldInfo.GRI).CampingSetting - 5) {
            if (BattlePlayerController(Owner) != none) BattlePlayerController(Owner).ServerShowMessage  (4);
          } 
          if (CampCounter == BattleGRI (WorldInfo.GRI).CampingSetting) {
            if (BattlePlayerController(Owner) != none) BattlePlayerController(Owner).ServerShowMessage (5);
          }
          if (CampCounter > BattleGRI (WorldInfo.GRI).CampingSetting) {
            TakeDamage(10, None, Location, vect(0,0,0), class'BattleDamageType_Camping');
            return;
          }
        } else {
          CampCounter = 0;
          CampLocation = C.Pawn.Location;
        }         

        // Lose 10 health per 5 seconds after round overtime
        if (RODN (0, 1) == 1 && Five) {
          TakeDamage (10, None, Location, vect(0,0,0), class'BattleDamageType_RoundOvertime');
        }
      }
    }
  }
}

function Freeze (BattlePRI BPRI)
{
  if (RemoteRole == ROLE_AutonomousProxy) {
    RemoteRole = ROLE_SimulatedProxy;
    bCanBeAutonomous = true;
  }
  BPRI.SpecialPlayerState = SPS_Frozen;
  StopFiring(); // This should work but sometimes seems to fail, no idea why; therefore ServerTimer also does this
  Health = 1;
  ShieldBeltArmor = 0;
  VestArmor = 0;
  ThighpadArmor = 0;
  HelmetArmor = 0;
  Acceleration = Vect (0,0,0);
  Velocity = Vect (0,0,0);
  setPhysics (PHYS_Falling);
  CustomGravityScaling = 2.0;
  SuperFreezeCounter = 10;
}

function Unfreeze (BattlePRI BPRI, optional int NewHealth=0)
{
  local int HealthCtr;
  local int HealthSum;
  local Controller CC;

  if (bCanBeAutonomous) {
    RemoteRole = ROLE_AutonomousProxy;
  }
  if (PlayerController(Owner) != none) PlayerController(Owner).ResetCameraMode();
  if (NewHealth == 0) {
    foreach WorldInfo.AllControllers(class'Controller', CC) {
      if (CC != none && CC.Pawn != None && CC.PlayerReplicationInfo != None && !CC.PlayerReplicationInfo.bOnlySpectator
        && BattlePRI (CC.PlayerReplicationInfo) != None
        && BattlePRI (CC.PlayerReplicationInfo).SpecialPlayerState == SPS_Alive) {
        HealthSum += CC.Pawn.Health;
        HealthCtr++;
      }
    }
  }
  if (HealthCtr > 0) {
    NewHealth = HealthSum / HealthCtr;
  }
  if (NewHealth < 10) {
    Health = 10;
  } else {
    Health = NewHealth;
  }
  BPRI.SpecialPlayerState = SPS_Alive; 
  ShieldBeltArmor = 0;
  VestArmor = 0;
  ThighpadArmor = 0;
  HelmetArmor = 0;
  CustomGravityScaling = 1.0;
  SuperFreezeCounter = 0;
}

simulated function StartFire(byte FireModeNum)  // Derived from UTPawn.uc
{
  // Disable weapons for frozen players, change the viewtarget instead
  if (BattlePRI (PlayerReplicationInfo) != None) {
    if (BattlePRI (PlayerReplicationInfo).SpecialPlayerState == SPS_Frozen) { 
      if (Owner != None && BattlePlayerController(Owner) != None) {
        if (FireModeNum==0) {
          BattlePlayerController(Owner).SpectateCycle();
        } else {
          BattlePlayerController(Owner).SpectateMe();
        }
      }
      return;
    }
  }

  if (bFeigningDeath) {
    FeignDeath();
  } else {
    Super.StartFire(FireModeNum);
  }
}

simulated function Tick (float deltaTime)
{
  if (BattleWeapon_ImpactHammer (Weapon) != None) {
    if (BattleGRI (WorldInfo.GRI).ImpactHammerSetting == 1) { // Standard
      MaxFallSpeed = 1250;
    } else if (BattleGRI (WorldInfo.GRI).ImpactHammerSetting == 0) { // Reduced Self Damage
      MaxFallSpeed = 1250;
    } else if (BattleGRI (WorldInfo.GRI).ImpactHammerSetting == 2) { // + Reduced Fall Damage
      MaxFallSpeed = 1500;
    } else if (BattleGRI (WorldInfo.GRI).ImpactHammerSetting == 3) { // + Boost
      MaxFallSpeed = 1750;
    } else if (BattleGRI (WorldInfo.GRI).ImpactHammerSetting == 4) { // + Ultra Boost
      MaxFallSpeed = 3000;
    }
    bSpecialFallDamage = true;
  } else {
    if (bSpecialFallDamage) {
      MaxFallSpeed = 1200;
      bSpecialFallDamage = false; // Only once, e.g. jump boots alter this as well
    }
  }
  if (WorldInfo.NetMode != NM_Client) { // Only on the server
    if (BattlePRI (PlayerReplicationInfo) != None) {
      if (BattlePRI (PlayerReplicationInfo).SpecialPlayerState == SPS_Frozen) { 
        if (SuperFreezeCounter-- >= 0) { // superfreeze for the first 10 ticks after being frozen
          Acceleration = Vect(0,0,0);
          Velocity = Vect(0,0,0);
          setPhysics (PHYS_Falling);
        }  
        if (Acceleration.X != 0 || Velocity.X != 0 || Acceleration.Y != 0 || Velocity.Y != 0) { // Ignore Z (elevators etc.)
          SuperFreezeMovesCounter++;
          if (SuperFreezeMovesCounter > 25) {
            Acceleration = Vect(0,0,0);
            Velocity = Vect(0,0,0);
            setPhysics (PHYS_Falling);
            SuperFreezeMovesCounter = 0;
          }
        }
      }
    }
  } else { // Only on the client
    if (BattlePRI (PlayerReplicationInfo) != None) {
      if (BattlePRI (PlayerReplicationInfo).SpecialPlayerState == SPS_Frozen) {
        if (Role == ROLE_AutonomousProxy) {
          Role = ROLE_SimulatedProxy;
          bCanBeAutonomous = true;
        }
      } else {
        if (Role == ROLE_SimulatedProxy && bCanBeAutonomous) {
          Role = ROLE_AutonomousProxy;
        }
      }
    }
  }
  Super.Tick (deltatime);
}

function bool Pawn_Died(Controller Killer, class<DamageType> damageType, vector HitLocation) // Derived from Pawn.uc
{
    local BattlePRI BPRI_InstigatedBy;
    local BattlePRI BPRI_Injured;

    local bool bKillMeAnyway;
	local SeqAct_Latent Action;

    // All life and death decisions are made here
    if (BattleFreezetagArena (WorldInfo.Game) != None) {
      bKillMeAnyway = false;
    } else {
      bKillMeAnyway = true;
    }
    if (WorldInfo.GRI.ElapsedTime < LastHurtBySlimeOrLava + 2) { // recently hurt by Lava or Slime
      bKillMeAnyway = true;
    }
    if (!bKillMeAnyway && BattlePRI (PlayerReplicationInfo) != None) {
      if (BattlePRI (PlayerReplicationInfo).Stats_DamageType2Index (damageType) == -1) { // E.g. falling or goo
        if (damageType != class'BattleDamageType_ReverseFriendlyFire') {
          bKillMeAnyway = true;
        }
      }
    }

    if (bKillMeanyway && BattlePRI (PlayerReplicationInfo) != None) {
      BattlePRI (PlayerReplicationInfo).SpecialPlayerState = SPS_VeryDead;    
    }

    // Cannot kill a frozen player
    if (!bKillMeAnyway && BattlePRI (PlayerReplicationInfo) != None) {
      if (BattlePRI (PlayerReplicationInfo).SpecialPlayerState == SPS_Frozen) { 
        Health = Max(Health, 1);
        return false;
      }
      bKillMeAnyway = false;
    } else {
      bKillMeAnyway = true; // Having no BattlePRI is a crime punishable by death
    }

	// allow the current killer to override with a different one for kill credit
	if ( Killer != None )
	{
		Killer = Killer.GetKillerController();
	}
	// ensure a valid damagetype
	if ( damageType == None )
	{
		damageType = class'DamageType';
	}
	// if already destroyed or level transition is occuring then ignore
	if ( bDeleteMe || WorldInfo.Game == None || WorldInfo.Game.bLevelChange )
	{
		return FALSE;
	}
	// if this is an environmental death then refer to the previous killer so that they receive credit (knocked into lava pits, etc)
	if ( DamageType.default.bCausedByWorld && (Killer == None || Killer == Controller) && LastHitBy != None )
	{
		Killer = LastHitBy;
	}
	// gameinfo hook to prevent deaths
	// WARNING - don't prevent bot suicides - they suicide when really needed
	if ( WorldInfo.Game.PreventDeath(self, Killer, damageType, HitLocation) )
	{
		Health = max(Health, 1);
		return false;
	}
	Health = Min(0, Health);

    if (bKillMeAnyway) {
	  // activate death events
  	  TriggerEventClass( class'SeqEvent_Death', self );
    } else {
      Freeze (BattlePRI (PlayerReplicationInfo));
    }

	// and abort any latent actions
	foreach LatentActions(Action)
	{
		Action.AbortFor(self);
	}
	LatentActions.Length = 0;

    if (bKillMeAnyway) {
        // notify the vehicle we are currently driving
        if ( DrivenVehicle != None )
        {
            Velocity = DrivenVehicle.Velocity;
            DrivenVehicle.DriverDied();
        }
        else if ( Weapon != None )
        {
            Weapon.HolderDied();
            ThrowActiveWeapon();
        }
    }

	// notify the gameinfo of the death
	if ( Controller != None )
	{
		BattleTeamArena(WorldInfo.Game).Special_Killed(Killer, Controller, self, damageType);
	}
	else
	{
		BattleTeamArena(WorldInfo.Game).Special_Killed(Killer, Controller(Owner), self, damageType);
	}

    if (!bKillMeAnyway) { // We keep the pawn alive if it is frozen
        BPRI_Injured = BattlePRI (Controller(Owner).PlayerReplicationInfo);
        if (Killer != None) BPRI_InstigatedBy = BattlePRI (Killer.PlayerReplicationInfo);
        if (BPRI_InstigatedBy != None) BPRI_InstigatedBy.Stats_RecordKill (BPRI_Injured, DamageType);
        Health = max(Health, 1);
        if (Killer != None) {
          BattleTeamArena(WorldInfo.Game).DetectRoundOver(Killer.PlayerReplicationInfo);
        } else {
          BattleTeamArena(WorldInfo.Game).DetectRoundOver(None);
        }
        return false;
    } else {   
        DrivenVehicle = None;
        // notify inventory manager
        if ( InvManager != None )
        {
            InvManager.OwnerEvent('died');
            // and destroy
            InvManager.Destroy();
            InvManager = None;
        }
        // push the corpse upward (@fixme - somebody please remove this?)
        Velocity.Z *= 1.3;
        // if this is a human player then force a replication update
        if ( IsHumanControlled() )
        {
            PlayerController(Controller).ForceDeathUpdate();
        }
        NetUpdateFrequency = Default.NetUpdateFrequency;
        PlayDying(DamageType, HitLocation);
        if (Killer != None) {
          BattleTeamArena(WorldInfo.Game).DetectRoundOver(Killer.PlayerReplicationInfo);
        } else {
          BattleTeamArena(WorldInfo.Game).DetectRoundOver(None);
        }
        return true;
    }
}

function bool Died(Controller Killer, class<DamageType> damageType, vector HitLocation) // Derived from UTPawn.uc
{
	if (Pawn_Died(Killer, DamageType, HitLocation))
	{
		StartFallImpactTime = WorldInfo.TimeSeconds;
		bCanPlayFallingImpacts=true;
		if(ArmsMesh[0] != none)
		{
			ArmsMesh[0].SetHidden(true);
			if(ArmsOverlay[0] != none)
			{
				ArmsOverlay[0].SetHidden(true);
			}
		}
		if(ArmsMesh[1] != none)
		{
			ArmsMesh[1].SetHidden(true);
			if(ArmsOverlay[1] != none)
			{
				ArmsOverlay[1].SetHidden(true);
			}
		}
		SetPawnAmbientSound(None);
		SetWeaponAmbientSound(None);
		return true;
	}
	return false;
}

simulated event PostRenderFor(PlayerController PC, Canvas Canvas, vector CameraPosition, vector CameraDir)
{
	local float TextXL, XL, YL, Dist;
	local vector ScreenLoc;
	local LinearColor TeamColor;
	local Color	TextColor;
	local string ScreenName;
	local UTWeapon Weap;
	local UTPlayerReplicationInfo PRI;
	local UTHUD HUD;

	screenLoc = Canvas.Project(Location + GetCollisionHeight()*vect(0,0,1));
	// make sure not clipped out
	if (screenLoc.X < 0 ||
		screenLoc.X >= Canvas.ClipX ||
		screenLoc.Y < 0 ||
		screenLoc.Y >= Canvas.ClipY)
	{
		return;
	}

	PRI = UTPlayerReplicationInfo(PlayerReplicationInfo);
	if (!WorldInfo.GRI.OnSameTeam(self, PC) )
	{
		// maybe change to action music if close enough
		if ( WorldInfo.TimeSeconds - LastPostRenderTraceTime > 0.5 )
		{
			if ( !UTPlayerController(PC).AlreadyInActionMusic() && (VSize(CameraPosition - Location) < VSize(PC.ViewTarget.Location - Location)) && !IsInvisible() )
			{
				// check whether close enough to crosshair
				if ( (Abs(screenLoc.X - 0.5*Canvas.ClipX) < 0.1 * Canvas.ClipX)
					&& (Abs(screenLoc.Y - 0.5*Canvas.ClipY) < 0.1 * Canvas.ClipY) )
				{
					// periodically make sure really visible using traces
					if ( FastTrace(Location, CameraPosition,, true)
									|| FastTrace(Location+GetCollisionHeight()*vect(0,0,1), CameraPosition,, true) )
					{
						UTPlayerController(PC).ClientMusicEvent(0);;
					}
				}
			}
			LastPostRenderTraceTime = WorldInfo.TimeSeconds + 0.2*FRand();
		}
		return;
	}

	// make sure not behind weapon
	if ( UTPawn(PC.Pawn) != None )
	{
		Weap = UTWeapon(UTPawn(PC.Pawn).Weapon);
		if ( (Weap != None) && Weap.CoversScreenSpace(screenLoc, Canvas) )
		{
			return;
		}
	}
	else if ( (UTVehicle_Hoverboard(PC.Pawn) != None) && UTVehicle_Hoverboard(PC.Pawn).CoversScreenSpace(screenLoc, Canvas) )
	{
		return;
	}

	// periodically make sure really visible using traces
	if ( WorldInfo.TimeSeconds - LastPostRenderTraceTime > 0.5 )
	{
		LastPostRenderTraceTime = WorldInfo.TimeSeconds + 0.2*FRand();
		bPostRenderTraceSucceeded = FastTrace(Location, CameraPosition)
									|| FastTrace(Location+GetCollisionHeight()*vect(0,0,1), CameraPosition);
	}
	if ( !bPostRenderTraceSucceeded )
	{
		return;
	}

	class'UTHUD'.Static.GetTeamColor( GetTeamNum(), TeamColor, TextColor);

	Dist = VSize(CameraPosition - Location);
	if ( Dist < TeamBeaconPlayerInfoMaxDist )
	{
		ScreenName = PlayerReplicationInfo.GetPlayerAlias()@" ("$BattlePRI(PlayerReplicationInfo).TotalHealth$")";
		Canvas.StrLen(ScreenName, TextXL, YL);
		XL = Max( TextXL, 24 * Canvas.ClipX/1024 * (1 + 2*Square((TeamBeaconPlayerInfoMaxDist-Dist)/TeamBeaconPlayerInfoMaxDist)));
	}
	else
	{
		XL = Canvas.ClipX * 16 * TeamBeaconPlayerInfoMaxDist/(Dist * 1024);
		YL = 0;
	}

	Class'UTHUD'.static.DrawBackground(ScreenLoc.X-0.7*XL,ScreenLoc.Y-1.8*YL,1.4*XL,1.9*YL, TeamColor, Canvas);

	if ( (PRI != None) && (Dist < TeamBeaconPlayerInfoMaxDist) )
	{
		Canvas.DrawColor = TextColor;
		Canvas.SetPos(ScreenLoc.X-0.5*TextXL,ScreenLoc.Y-1.2*YL);
		Canvas.DrawTextClipped(ScreenName, true);
	}

	HUD = UTHUD(PC.MyHUD);
	if ( (HUD != None) && !HUD.bCrosshairOnFriendly
		&& (Abs(screenLoc.X - 0.5*Canvas.ClipX) < 0.1 * Canvas.ClipX)
		&& (screenLoc.Y <= 0.5*Canvas.ClipY) )
	{
		// check if top to bottom crosses center of screen
		screenLoc = Canvas.Project(Location - GetCollisionHeight()*vect(0,0,1));
		if ( screenLoc.Y >= 0.5*Canvas.ClipY )
		{
			HUD.bCrosshairOnFriendly = true;
		}
	}
}

reliable server function ServerFeignDeath() { }
exec simulated function FeignDeath() { }
simulated function PlayFeignDeath() { }

defaultproperties
{
  Name="BattlePawn"
}