//-----------------------------------------------------------
//  Game Logger Main Module
//  Revision: v4.0 RC3
//-----------------------------------------------------------
//  Game Stats Class that hooks events and writes them down
//  in a log
//-----------------------------------------------------------
//  by Michael "_Lynx" Sokolkov  2004-2005
//-----------------------------------------------------------
class EGameStats extends GameStats config(GameLogger);

var GameLoggerRules G;
var MessagelogHandler MessageLogger;

// ============================================================================
// Configurables. These are set from the MutGameLogger.
// ============================================================================

var globalconfig bool bUseColoredNames;         // if player's name will have his team's color
var globalconfig bool bUseColoredTeamNames;     // Team name will have appropriate color
var globalconfig bool bShowTeamScoreEvents;     // if the teams' scoring events will be logged
var globalconfig bool bShowConnectsDisconnects; // Log connects disconnects for players
var globalconfig bool bLogOtherPicks;           // Log picking health, armor, udamage, etc
var globalconfig bool bLogWeaponsPicks;         // log picking weapons
var globalconfig bool bFirstRun;                // ??? may be obsolete
var globalconfig bool bShowSStatus;             // Show special status?
var globalconfig bool bIgnoreWarmupEvents;      // If the logger should ignore warm up events (only for GameLogger's log)
var globalconfig bool bReportGameEvents;        //
var globalconfig bool bUseGraphics;             // If true, the HTMLwrapper writes IMG tags
var globalconfig bool bWSLenabled;              // If true, world stats logger is enabled
var globalconfig bool bLogTalks;                // Should logger write players' talks?
var globalconfig bool bDebug;                   // Enable writimg debug messages to the log?
var globalconfig bool bTurnOff;                 // true, if Logger used only as host for other plugins, (f.e. with GameLoggerCTFsummary)
var globalconfig bool bDuelMode;                // Is it a 1x1 match?
var globalconfig bool bReportRespawns;          // not yet implemented
var globalconfig bool bExtendedPlayerStats;     //if TTM or UTComp is present, get their weapon stats also.
var globalconfig bool bReportHeavyDamage;       // should logger report about inflicting heavy damage
var globalconfig bool bBroadcastMatchInfo;      // doesn't seem to work yet. need moving elsewhere.
var globalconfig bool bExportDetailedStats;     // if true, logger also dumps the additional stats (the one form f3 screen) after the scoreboard
var globalconfig bool bColoredMessages;
var globalconfig bool bDontLogKills;
var globalconfig bool bSortOffOnlySpecs;

var globalconfig int HeavyDamageReportMode,     // 0 - don't log; 1 - log but only if vicitim stays alive; 2 - always;
                     PlayerScoreEventsMode;     // 0 - don't log; 1 - log all but kills; 2 - any;

var globalconfig string serverIP;

// ============================================================================
// Plugin systems. Note, that these will be set from MutGameLogger
// ============================================================================

var() globalconfig array< class<GameLoggerWeaponDamagePlugin> > DamageTypePluginClasses;
var() globalconfig array< class<GameLoggerExtraStatsPlugin> >   ExtraStatsPluginClasses;
var() globalconfig array< class<GameLoggerReportPlugin> >       ReportPluginClasses;
var() globalconfig array< class<GameRules> >                    StatsGameRulesClasses;    // Not implemented yet;

var array<GameLoggerExtraStatsPlugin>      ExtraStatsPlugins;
var array<GameLoggerWeaponDamagePlugin>    DamageTypePlugins;
var array<GameLoggerReportPlugin>          ReportPlugins;
var array<GameRules>                       StatsGameRules;

// ============================================================================
// HTML-wrapper
// ============================================================================

var() globalconfig class<GameLoggerHTMLwrapper> myHTMLwrapperClass;

var GameLoggerHTMLwrapper myHTMLwrapper;

// ============================================================================
// Localization
// ============================================================================
// Game events
// ============================================================================
var localized string matchplayed,
                     Player,
                     Ping,
                     PL,
                     RulesString,
                     GameStartedMsg,
                     EnteredTheGameMsg,
                     LeftTheGameMsg,
                     GameEndedMsg,
                     ByCompletingObjective,
                     ByCapturingFlag,
                     CapturesTheFlag,
                     FlagIsCaptured,
                     PerformsSuicide,
                     heavyselfdamageM,
                     heavyselfdamageF,
                     damagestring,
                     bystring,
                     leavinghim,
                     leavingher,
                     damagesstring,
                     ByHealingPowerNode,
                     DestroyedEnemyPowerCore,
                     ProtectingTeammate,
                     ByKillingAnEnemy,
                     ByKillingAnEnemyTeam, //Team means this is a team message;
                     KillsTeammate, //for team
                     KilledATeammate, //for player
                     WinsASRound,
                     ScoredGoal,
                     ByScoringGoal,
                     CarriedBall,
                     ByCarryingBall,
                     ByAssisting,
                     DOMScores,
                     DOMByScoring,
                     NameChange,
                     FlagTaken,
                     FlagReturnedTimeout,
                     FlagDropped,
                     FlagPicked,
                     FlagReturned,
                     FlagCaptured,
                     TeamChanged,
                     CompletedLastObjective,
                     AttackingTeamWon,
                     DefendingTeamWon,
                     NewRoundStarted,
                     NewRoundStarted2,
                     CompletedObjective,
                     DestroyedVehicle,
                     BallDropped,
                     BallTaken,
                     BallPicked,
                     BallReturnedTimeout,
                     RedTeam,
                     Scores, // player/team SCORES ## point(s)
                     Loses, // the same
                     BlueTeam,
                     LastSecondSaveBy,
                     GibbedInTranslocation,
                     KilledTypingVictim,
                     DrawsFirstBlood,
                     ArenaName,
                     ServerName,
                     MutatorsString,
                     RunsBoosterCombo,
                     RunsInvisCombo,
                     RunsBerserkCombo,
                     RunsSpeedCombo,
                     RunsCrateCombo,
                     RunsMinimeCombo,
                     RespawnedAt,
                     Admin,
                     Point,
                     Points,
                     Points2; //for russian; for INT the same as "Points"

// ============================================================================
// Weapon names for HDamage events, kills and pickups
// ============================================================================
var localized string WeaponNames[18];
var localized string PickedWeaponNames[18];
// ============================================================================
// Damagetype names;
// ============================================================================
var localized string Combo,
                     headshot,
                     headsuffix,
                     burned,
                     corroded,
                     crushed,
                     drowned,
                     fell,
                     felllava,
                     Sburned,
                     Scorroded,
                     Scrushed,
                     Sdrowned,
                     Sfell,
                     Sfelllava,
                     whiletypingM,
                     whiletypingF;

var localized string RaptorMissile,
                     RaptorPlasma,
                     HBNDSideTurretSec,
                     HBNDBackTurret,
                     MantaPlasma,
                     LinkTurretPlasma,
                     LinkTurretBeam,
                     IonTankCannon,
                     IonCannonShot,
                     LeviBlast,
                     LeviTurret,
                     MiniTurret,
                     VehicleExplosion,
                     BlownVehicleRK,
                     Accident,
                     GoliathTurret,
                     ScorpBlade,
                     ScorpNet,
                     HBNDSideTurretCombo,
                     HBNDSideTurretPri,
                     HFighterLaser,
                     SFighterLaser,
                     HFighterMissile,
                     SFighterMissile,
                     GoliathCannon,
                     ShockTurret,
                     RaptorRK,
                     MantaHeadClip,
                     IonTankRK,
                     LeviRK,
                     RaptorPK,
                     MantaPK,
                     IonTankPK,
                     LeviPK,
                     HBND_PK,
                     ScorpPK,
                     GoliathPK,
                     HBND_RK,
                     ScorpRK,
                     GoliathRK;


var bool bReversedCombo;// specially for transaltions, should be true for english.
// ============================================================================
// some sentence construction related strings
// ============================================================================
var localized string firetype[2]; //0 = primary, 1 = secondary
var localized string killsstring,
                     suicidestring,
                     with,
                     Awith,
                     NATprefix,
                     push,
                     pushsuffix,
                     crush,
                     ATsuffix,
                     picks,
                     lands_on,
                     runsover,
                     clips,
                     Gets,
                     Keeps;
// ============================================================================
// Armor and health, and other pickups names for pickup logging
// ============================================================================
var localized string ArmorNames[2];
var localized string HealthNames[2];
var localized string UDamageName;

// ============================================================================
// Onslaught events
// ============================================================================
var localized string ONSPowerNode_constructed,
                     ONSPowerNode_destroyeddueisolation,
                     ONSPowerNode_destroyed,
                     ONSPowerNode_beingconstructed,
                     ONSPowerNode_disabled,
                     ONSCoreDestroyed,
                     ONSPowerNodeName;

// ============================================================================
// Internal Variables
// ============================================================================

// ============================================================================
// Boolean vars
// ============================================================================
var bool IsTeamGame;             // is the current gametype is team-based
var bool bUseName;
var bool bTTMIsPresent;          //
var bool bUTCompIsPresent;       // Used to detect overtimes.
var bool bONS;                   // if ONS, is true and so, the logger checks node state changes
var bool bCTF;                   // if CTF, is true and so, the logger checks flags state changes
var bool bClosedReport;          // To avoid writing any strings after the report table is closed
var bool bMatchStarted;			     // Used to start counting time from 0, after the match started.
var bool bUTCompWarmUpStarted;	 // Used to detect first call of StartMatch()

// ============================================================================
// String vars
// ============================================================================
var string Player1;           // Player 1 name for duel mode
var string Player2;           // Player 2 name for duel mode
var string winner;            // winner string - used for the endgame "%p/%t wins the match"
var string TempPickupString;  // used to prevent pickup spamming
var const string Version;     // version of the GameLogger

// ============================================================================
// Float vars... er... var
// ============================================================================
var float MatchStartTime;

// ===========================================================================
// Structs
// ===========================================================================
struct LastSpreeOwner{                     // Last spree owner
  var PlayerReplicationInfo SpreeOwner;    // Last spree owner's PRI
  var string SpreeLevel;                   // his spree level
};

struct PowerNodes{                         // PowerNodes
  var ONSPowerNodeNeutral thisNode;        // powernode
  var byte oldstate;                       // it's state on prev check, compared with the current
};

struct Flags{                              // CTF Flags
  var CTFFlag thisFlag;                    // this Flag
  var bool bLastSecond;                    // was it last second?
};

// ============================================================================
// Arrays
// ============================================================================
var array<PowerNodes> PowerNodesArray;     // array of PowerNodes structs
var array<Flags> FlagsArray;               // array of  Flags structs

// ============================================================================
// Classes references
// ============================================================================
var LastSpreeOwner LSO;                    // needs fixing should be array;
var WSLlogger WSL;                         // reference to Epics world stats logger
//var MutStatsExporter Loader;               // Mutator, loading the GL

// ===========================================================================
// PreBeginPlay()
// ===========================================================================
function PreBeginPlay()
{
  local int i;
  local int l;

  local GameLoggerReportPlugin TestPlugin;
  local MessageLogHandler MessageHook;
  local GameLoggerRules PickupHook;

  local bool bRulesHookAlreadyspawned;

  Super.PreBeginPlay();

  bLocalLog   = True;
  IsTeamGame  = Level.Game.bTeamGame;
  bUseName    = True;

  // Beware of an inventive fool that'll go to the ini and change the values
  // to the improper ones manually.
  // Fool-proof section

  // some illegal value is set (out of 0-2 range), resetting to default
  if (PlayerScoreEventsMode > 2 || PlayerScoreEventsMode < 0){
	  PlayerScoreEventsMode = 1;
	}
	if (HeavyDamageReportMode > 2 || HeavyDamageReportMode < 0){
	  HeavyDamageReportMode = 0;
	}

	if (ServerIP != "" && Left(ServerIP, 9) ~= "ut2004://")
	  ServerIP = Repl(ServerIP, "ut2004://", "");

//  if (Level.Game.ClassIsChildOf(Level.Game, class'TeamGame'))
//    TeamGame(Level.Game).bScoreVictimsTarget = True;

  //Initializing World Stats Logging support
  if (bWSLenabled){
    WSL = spawn(class'WSLlogger');
    WSL.GRI = Level.Game.GameReplicationInfo;
  }

  // spawning Rules class to hook in pickups and damage events;
  G = spawn(class'GameLoggerRules');

  if ( Level.Game.GameRulesModifiers == None )
    Level.Game.GameRulesModifiers = G;
  else
    Level.Game.GameRulesModifiers.AddGameRules(G);

  // spawning our broadcast handler, to log players' talks
  MessageLogger = Spawn(class'MessageLogHandler');

  //Initializing Message hook
  foreach AllActors(class'MessageLogHandler',MessageHook)
  {
    if (!MessageHook.bIsInitialized){
    if ( Level.Game.BroadcastHandler != None )
      Level.Game.BroadcastHandler.RegisterBroadcastHandler(MessageHook);
      MessageHook.MyStatsActor   = self;
      MessageHook.bIsInitialized = True;
    }
  }

  // Initializing pickHook for
  foreach AllActors(class'GameLoggerRules',PickupHook)
  {
    if (!PickupHook.bIsInitialized && !bRulesHookAlreadyspawned){
      PickupHook.MyStats        = self;
      PickupHook.bIsInitialized = True;
      bRulesHookAlreadyspawned  = True;
    }
  }

  // Initializing Damagetypes plugins
  DamageTypePlugins.Length = DamageTypePluginClasses.Length;

  Log("// ======================================================================");

  for (i = 0; i < DamageTypePluginClasses.Length; i++){
    DamageTypePlugins[i] = spawn(DamageTypePluginClasses[i]);
    Log("// GameLogger Damagetype Plugin System: Added "$DamageTypePlugins[l].FriendlyName@"by"@DamageTypePlugins[l].Author);
    Log("// GameLogger Damagetype Plugin System: Description: "$DamageTypePlugins[l].Description);
    Log("// GameLogger Damagetype Plugin System: version: "$DamageTypePlugins[l].Version);
  }

  // Initializing Report plugins
  // a bit more tricky - checks if this plugin is designed to support the
  // current gametype, to avoid unnecessary accessed Nones.
  ReportPlugins.Length = 0;

  for (i = 0; i < ReportPluginClasses.Length; i++){

    // I know that the this function can be called without spawning the class,
		// if it is declared as 'static'. I haven't done so because if the plugins
		// fail to spawn because it requires some packages that are missing, it is
		// better if this happen on a test load, not when adding to array, as this
		// will leave an empty entry in plugins array, causing an Accessed None each
		// time EGameStats passes a function call to all report plugins.
    TestPlugin = spawn(ReportPluginClasses[i]);

    if (TestPlugin != None && testPlugin.ForThisGametype(Level.Game.GameName)){

      l = ReportPlugins.Length;
      ReportPlugins.Length = l + 1;

      ReportPlugins[l] = spawn(ReportPluginClasses[i]);
      ReportPlugins[l].BaseLogger   = self;
      ReportPlugins[l].InitializePlugin();
      Log("// GameLogger Report Plugin System: Added "$ReportPlugins[l].FriendlyName@"by"@ReportPlugins[l].Author);
      Log("// GameLogger Report Plugin System: Description: "$ReportPlugins[l].Description);
      Log("// GameLogger Report Plugin System: version: "$ReportPlugins[l].Version);
    }
    TestPlugin.Destroyed();
  }

  ExtraStatsPlugins.Length = ExtraStatsPluginClasses.Length;

  for (i = 0; i < ExtraStatsPluginClasses.Length; i++){
    ExtraStatsPlugins[i] = spawn(ExtraStatsPluginClasses[i]);
    ExtraStatsPlugins[i].BaseLogger   = self;
    ExtraStatsPlugins[i].bInitialized = true;
    Log("// GameLogger Extra Stats Plugin System: Added "$ExtraStatsPlugins[i].FriendlyName@"by"@ExtraStatsPlugins[l].Author);
    Log("// GameLogger Extra Stats Plugin System: Description: "$ExtraStatsPlugins[i].Description);
    Log("// GameLogger Extra Stats Plugin System: version: "$ExtraStatsPlugins[i].Version);
  }
  Log("// ======================================================================");


  // Initializing HTML wrapper
  myHTMLwrapper = spawn(myHTMLwrapperClass,self);

  if (myHTMLwrapper == None){
    Warn("GameLogger: No HTML wrapper is set! Using default one!");
  	myHTMLwrapper = spawn(class'GameLogger.GameLoggerHTMLwrapper',self);
  }

  myHTMLwrapper.BaseLogger = self;
}

// ===================================================
// PostBeginPlay()
//
// Creates an array of active power nodes to track
// their state afterwards
// ===================================================

function PostBeginPlay()
{
  local ONSPowerNodeNeutral NewNode;
  local int l;

  Super.PostBeginPlay();

  if (Level.Game.IsA('ONSOnslaughtGame')){
    bONS = True;
    if (bDebug){
    	 Log("Gametype is :"@Level.Game.GameName$", enabling ONS support");
    }
    foreach AllActors(class'ONSPowerNodeNeutral', NewNode)
    {
       if (NewNode != None && NewNode.CoreStage != 255){
         l = PowerNodesArray.Length;
         PowerNodesArray.Length = l + 1;
         PowerNodesArray[l].thisNode = NewNode;
         PowerNodesArray[l].oldstate = NewNode.CoreStage;
         if (bDebug){
        	 Log("Found a powernode:"@NewNode.NodeNum);
        	 Log("Added to nodes array under number"@l);
         }
       }
    }
  }

  SetTimer(0.2, True);
}

// ============================================================================
// Timer
//
// Being run once a second. Does several tasks:
// 1. Checks the PowerNodes
// 2. Checks the CTFFlags to get the last second save. Needs rewriting, as
//    current approach does not work.
// ============================================================================

function Timer()
{
  local int i;
  local ONSPowerNodeNeutral testNode;
//  local CTFFlag testFlag;

  if (bONS){
    foreach VisibleCollidingActors(class'ONSPowerNodeNeutral', testNode, 512000)
    {
      for(i = 0; i < PowerNodesArray.Length; i++)
        if (PowerNodesArray[i].thisNode == testNode){
          if (PowerNodesArray[i].OldState != testNode.CoreStage){
            if (bDebug){
        	 		Log("PowerNode"@testNode.NodeNum@"state has changed from"@PowerNodesArray[i].OldState@"to"@testNode.CoreStage);
		         }
             NodeStateChange(testNode, PowerNodesArray[i].OldState);
             PowerNodesArray[i].OldState = testNode.CoreStage;
          }
        }
    }
  }

  Super.Timer();
}

event Destroyed()
{
  Shutdown();
  Level.Game.Broadcast(self, "Game Log "$GetLogFilename()$" Saved", 'GameLoggerStatus');
  Super.Destroyed();
}


// ============================================================================
//  Utility Functions returning strings
// ============================================================================
// This function recives seconds passed from
// the start of the match and converst them
// to the hours minutes & seconds
// ============================================================================
function String FormatTime( int Seconds )
{
  local int Minutes, Hours, fixedSeconds;
  local String Time;

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// That's the fix for the Level.TimeDilation
// which is always equal to 1.1, so for the
// game, 20 minutes real time are equal to 22
// of ingame time. This resulte in showing
// that a 20 min game lasted 22 mins.
  fixedSeconds = int(Seconds / Level.TimeDilation);
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

  Seconds = fixedSeconds;

  if( Seconds > 3600 )
  {
    Hours = Seconds / 3600;
    Seconds -= Hours * 3600;

    Time = Hours$":";
  }
  Minutes = Seconds / 60;
  Seconds -= Minutes * 60;

  if( Minutes >= 10 )
    Time = Time $ Minutes $ ":";
  else
    Time = Time $ "0" $ Minutes $ ":";

  if( Seconds >= 10 )
    Time = Time $ Seconds;
  else
    Time = Time $ "0" $ Seconds;

  return Time;
}


// ============================================================================
// TimeStamp
//
// Returns the string with current time
// from the start of the match
// ============================================================================
function string TimeStamp()
{
  local string seconds;
  local int SecInt;

  SecInt = int(Level.TimeSeconds);

  // if match started, substract the time that passed since the level loaded.
  if (bMatchStarted){
      SecInt = int(Level.TimeSeconds - MatchStartTime);
  }
  seconds = FormatTime(SecInt);

  return seconds;
}

// ============================================================================
// CCName (colored character(?) name)
//
// Returns the name of the player from given
// PRI and if required adds status tag and
// draws name in appropriate team color
// ============================================================================
function string CCName (PlayerReplicationInfo thisPRI)
{
  local string result;

  if (thisPRI == None){
    result = "";
    return result;
  }

  if (bUseColoredNames && IsTeamGame){
    result = myHTMLwrapper.ColorPlayerName(thisPRI.PlayerName, thisPRI.Team.TeamIndex);
  }
  else result = thisPRI.PlayerName;

  if (thisPRI.bAdmin && bShowSStatus)
    result = result@"[ADMIN]";
  if (thisPRI.bBot && bShowSStatus)
    result = result@"[BOT]";
  if (thisPRI.bIsSpectator && bShowSStatus)
    result = result@"[SPECTATOR]";

  return result;
}


// ============================================================================
// CTName (colored team name)
//
// Returns the name of the team by given team
// index and if required adds appropriate
// name color tags
// ============================================================================
function string CTName (int thisTeamIndex, optional bool bNoColor)
{
  local string result,
               BlueTeamName,
               RedTeamName;

  BlueTeamName = GRI.Teams[1].TeamName;
  RedTeamName = GRI.Teams[0].TeamName;

  if (bUseColoredTeamNames && !bNoColor)
  {
    if (thisTeamIndex == 1){
      if ( BlueTeamName == "" || BlueTeamName == "Team" )
        result = myHTMLwrapper.ColorTeamName(BlueTeam, 1);
      else result = myHTMLwrapper.ColorTeamName(BlueTeamName, 1);
    }
    else if (thisTeamIndex == 0){
      if ( RedTeamName == "" || RedTeamName == "Team" )
        result = myHTMLwrapper.ColorTeamName(redTeam, 0);
      else result = myHTMLwrapper.ColorTeamName(RedTeamName, 0);
    }
  }
  else {
    if (thisTeamIndex == 1){
      if ( BlueTeamName == "" || BlueTeamName == "Team" )
        result = BlueTeam;
      else result = BlueTeamName;
    }
    else if (thisTeamIndex == 0){
      if ( RedTeamName == "" || RedTeamName == "Team" )
        result = RedTeam;
      else result = RedTeamName;
    }
  }

  return result;
}


// ============================================================================
// CheckForDamageInPlugins
//
// Looks through the damage type plugins (subclasses of
// GameLoggerWeaponDamagePlugin) and returns the appropriate action string if
// the input damagetype matches one in plugin
// ============================================================================
function string CheckForDamageInPlugins(class<DamageType> testDamageType){
  local string result;
  local int i, z;

  for (i = 0; i < DamageTypePlugins.length; i++ ){ //moving through plugins array
    if (DamageTypePlugins[i] != None){//if the current plugin entry is not empty...
      for (z = 0; z < DamageTypePlugins[i].ExtraDamageType.Length; z++ ){//...look through it's damagetype array
        if (testDamageType == DamageTypePlugins[i].ExtraDamageType[z].thisDamageType) //and if the damagetype passed is the one named in array
          result = DamageTypePlugins[i].ExtraDamageType[z].thisAction;
      }
    }
  }

//  if (result == ""){
//
//  }
  return result;
}
// ============================================================================
// GetExtraStatsFromPlugins
//
// Looks through the stats plugins (subclasses of
// GameLoggerExtraStatsPlugin) and makes them dump their stats buffer
// ============================================================================
function GetExtraStatsFromPlugins(PlayerReplicationInfo PRIs, string TDstyle){
  local int i;

  for (i = 0; i < ExtraStatsPlugins.Length; i++){
    ExtraStatsPlugins[i].GimmeStats(PRIs, Level.Game.GameReplicationInfo, TDStyle);
  }
}

// ============================================================================
// Classify Event
//
// Classifies the game event by it's name
// and returns the appropriate action string
// ============================================================================
function string ClassifyEvent (string ED, optional string sw)
{
 local string result;

 if ( ED ~= "ObjectiveScore"){
   result = ByCompletingObjective;
 }
 else if ( ED ~= "flag_cap_final"){
   result = ByCapturingFlag;
 }
 else if ( ED ~= "flag_cap"){
   result = CapturesTheFlag;
 }
 else if ( ED ~= "flag_captured"){
   result = FlagIsCaptured;
 }
 else if ( ED ~= "self_frag"){
   result = PerformsSuicide;
 }
 else if ( ED ~= "heal_powernode"){
   result = ByHealingPowerNode;
 }
 else if ( ED ~= "enemy_core_destroyed"){
   result = DestroyedEnemyPowerCore;
 }
 else if ( ED ~= "team_protect_frag"){
   result = ProtectingTeammate;
 }
 else if ( ED ~= "tdm_frag"){
   result = ByKillingAnEnemyTeam;
   }
 else if ( ED ~= "frag"){
   result = ByKillingAnEnemy;
 }
 else if (sw == "player" && ED ~= "team_frag"){
   result = KilledATeammate;
 }
 else if (sw == "team" && ED ~= "team_frag"){
   result = KillsTeammate;
 }
 else if ( ED ~= "pair_of_round_winner"){
   result = WinsASRound;
 }
 else if ( ED ~= "ball_tossed"){
   result = ScoredGoal;
 }
 else if ( ED ~= "ball_thrown_final"){
   result = ByScoringGoal;
 }
 else if ( ED ~= "ball_carried"){
   result = CarriedBall;
 }
 else if ( ED ~= "ball_cap_final"){
   result = ByCarryingBall;
 }
 else if ( ED ~= "ball_score_assist"){
   result = ByAssisting;
 }
 else if ( ED ~= "dom_teamscore"){
   result = DOMScores;
 }
 else if ( ED ~= "dom_score"){
   result = DOMByScoring;
 }
 else if (sw == "team" && ED ~= "team_frag"){
   result = KillsTeammate;
 }
 else {
   result = "";
 }

 return result;
}


// ============================================================================
// DecodeSEDesc
//
// Classifies the special event by it's name
// and returns the appropriate action string
//=============================================================================
function string DecodeSEDesc(string ED, PlayerReplicationInfo Who)
{
  local string result,
               tempresult;
  local bool bMultiKill,
             bCanKeepHS,
             bKeepHS;

 if ( ED ~= "translocate_gib" ){
   result = GibbedInTranslocation;
 }
 else if ( ED ~= "type_kill" ){
   result = KilledTypingVictim;
 }
 else if ( ED ~= "first_blood" ){
   result = DrawsFirstBlood;
 }
 else if ( ED ~= "spree_1" ){
   result = class'KillingSpreeMessage'.default.SpreeNote[0];
 }
 else if ( ED ~= "spree_2" ){
   result = class'KillingSpreeMessage'.default.SpreeNote[1];
 }
 else if ( ED ~= "spree_3" ){
   result = class'KillingSpreeMessage'.default.SpreeNote[2];
 }
 else if ( ED ~= "spree_4" ){
   result = class'KillingSpreeMessage'.default.SpreeNote[3];
 }
 else if ( ED ~= "spree_5" ){
   result = class'KillingSpreeMessage'.default.SpreeNote[4];
 }
 else if ( ED ~= "spree_6" ){
   result = class'KillingSpreeMessage'.default.SpreeNote[5];
 }
 else if ( ED ~= "multikill_1" ){
   result = class'MultiKillMessage'.default.KillString[0];
   bMultiKill=true;
 }
 else if ( ED ~= "multikill_2" ){
   result = class'MultiKillMessage'.default.KillString[1];
   bMultiKill=true;
 }
 else if ( ED ~= "multikill_3" ){
   result = class'MultiKillMessage'.default.KillString[2];
   bMultiKill=true;
 }
 else if ( ED ~= "multikill_4" ){
   result = class'MultiKillMessage'.default.KillString[3];
   bMultiKill=true;
 }
 else if ( ED ~= "multikill_5" ){
   result = class'MultiKillMessage'.default.KillString[4];
   bMultiKill=true;
 }
 else if ( ED ~= "multikill_6" ){
   result = "L U D I C R O U S   K I L L!!!"; // the default is just LUDICROUS,
   bMultiKill=true;
 }
 else if ( ED ~= "multikill_7" ){
   result = class'MultiKillMessage'.default.KillString[6];
   bMultiKill=true;
 }
 else if ( ED ~= "XGame.ComboDefensive"){
   result = RunsBoosterCombo;
 }
 else if ( ED ~= "XGame.ComboInvis"){
   result = RunsInvisCombo;
 }
 else if ( ED ~= "XGame.ComboBerserk"){
   result = RunsBerserkCombo;
 }
 else if ( ED ~= "XGame.ComboSpeed"){
   result = RunsSpeedCombo;
 }
 else if ( ED ~= "XGame.ComboCrate"){
   result = RunsCrateCombo;
 }
 else if ( ED ~= "XGame.MiniMe"){
   result = RunsMinimeCombo;
 }
 else if (Left(ED,9) ~= "multikill_" &&
									ED != "multikill_1" &&
									ED != "multikill_2" &&
									ED != "multikill_3" &&
									ED != "multikill_4" &&
									ED != "multikill_5" &&
									ED != "multikill_6" &&
									ED != "multikill_7") {
   result = class'MultiKillMessage'.default.KillString[6];
   bCanKeepHS=True;
   bMultiKill=True;
 }

 tempresult = result;

 if (bMultiKill){
   if (LSO.SpreeOwner == Who && LSO.SpreeLevel == result && bCanKeepHS) //player holds Holy Shit level
     bKeepHS = True;
     LSO.SpreeOwner = Who;
     LSO.SpreeLevel = tempresult;
   if (bKeepHS)
     return keeps@class'MultiKillMessage'.default.KillString[6];
   else
     return gets@result;
  }

  else return result;
}


// ============================================================================
//  ClassifyGEvent
//
// Returns the appropriate string for give game event
// ============================================================================

function string ClassifyGEvent (string ED, string Misc)
{
  local string result;

  if ( ED ~= "flag_taken"){
    result = FlagTaken;
  }
  else if ( ED ~= "flag_returned_timeout"){
    result = myHTMLwrapper.B(CTName(int(Misc)))@FlagReturnedTimeout;
  }
  else if ( ED ~= "flag_dropped"){
    result = FlagDropped;
  }
  else if ( ED ~= "flag_returned"){
    result = FlagReturned;
  }
  else if ( ED ~= "flag_captured"){
    result = FlagCaptured;
  }
  else if ( ED ~= "flag_pickup"){
    result = FlagPicked;
  }
  else if ( ED ~= "TeamChange"){
    result = TeamChanged@myHTMLwrapper.B(CTName(int(Misc)));
  }
  else if ( ED ~= "EndRound_Trophy"){
    result = CompletedLastObjective;bUseName=False;
  }
  else if ( ED ~= "AS_attackers_win"){
    result = AttackingTeamWon;
  }
  else if ( ED ~= "AS_defenders_win"){
    result = DefendingTeamWon;
  }
  else if ( ED ~= "AS_BeginRound"){
    result = myHTMLwrapper.B(NewRoundStarted);
  }
  else if ( ED ~= "ObjectiveCompleted_Trophy"){
    result = CompletedObjective;
  }
  else if ( ED ~= "VehicleDestroyed_Trophy"){
    result = DestroyedVehicle;
  }
  else if ( ED ~= "bomb_dropped"){
    result = BallDropped;
  }
  else if ( ED ~= "bomb_taken"){
    result = BallTaken;
  }
  else if ( ED ~= "bomb_pickup"){
    result = BallPicked;
  }
  else if ( ED ~= "bomb_returned_timeout"){
    result = BallReturnedTimeout;
  }
  else if ( ED ~= "NameChange"){
    result = NameChange@myHTMLwrapper.B(Misc);
  }
  else {
    result = "";
  }

  return result;
}

// ============================================================================
// PointsString
//
// Localization support function. In some languages word "point" can be in
// different form depending on the amount of points. f.e. in Russian
// ============================================================================

function string PointsString( float amount, optional string Desc)
{
  local string result;
  local int IntAmount;

  IntAmount = int(Amount);

  if (Desc == "team_frag"){
    if ( amount == 1 )
      result = Loses@IntAmount@Point;
    else if (amount == 2 || amount == 3 || amount == 4)
      result = Loses@IntAmount@Points;
    else
      result = Loses@IntAmount@Points2;
   }
   else{
     if ( IntAmount  > 0 ){
       if ( amount == 1 )
         result = Scores@IntAmount@Point;
       else if (amount == 2 || amount == 3 || amount == 4)
       result = Scores@IntAmount@Points;
       else
       result = Scores@IntAmount@Points2;
     }
     else if ( IntAmount  < 0 ){
       if ( amount == -1 )
         result = Loses@-IntAmount@Point;
       else if (amount == -2 || amount == -3 || amount == -4)
         result = Loses@-IntAmount@Points;
       else
         result = Loses@-IntAmount@Points2;}
    }

 return result;
}

// ============================================================================
// GL (Get Location)
//
// returns location of player Who
// ============================================================================

function string GL(PlayerReplicationInfo Who)
{
  return Who.GetLocationName();
}

// ============================================================================
// InsertBR
//
// parses the given string and replaces the "|" symbol which is used for line
// breaks in UT2003/UT2004 GUI with HTML line break tag
// ============================================================================

function string InsertBR(string D)
{
 local string result;

 result = D;
 replacetext(result, "|", "<br>");

 return result;
}

// ============================================================================
// WriteHeader
//
// Calls WriteH in wrapper.
// ============================================================================
function WriteHeader (string GameType, string MapName)
{
  myHTMLwrapper.DoWriteHeader(GameType, Level.Title);  // Map name nearly always returns Untitled
  Log("// ==================================================");
  Log("// GameLogger v"$version@"started");
  Log("// The settings are:");
  Log("// bUseColoredNames is"@bUseColoredNames);
  Log("// bUseColoredTeamNames is"@bUseColoredTeamNames);
  Log("// PlayerScoreEventsMode is "@PlayerScoreEventsMode);
  Log("// bShowTeamScoreEvents is"@bShowTeamScoreEvents);
  Log("// bShowConnectsDisconnects"@bShowConnectsDisconnects);
  LOg("// bLogOtherPicks is"@bLogOtherPicks);
  log("// bLogWeaponsPicks is"@bLogWeaponsPicks);
  log("// bShowSStatus is"@bShowSStatus);
  Log("// bReportGameEvents is"@bReportGameEvents);
  Log("// bUseGraphics is"@bUseGraphics);
  Log("// bWSLenabled is"@bWSLenabled);
  Log("// bLogTalks is"@bLogTalks);
  Log("// HeavyDamageReportMode is"@HeavyDamageReportMode);
  Log("// bLogTalks is"@bDuelMode);
  Log("// ==================================================");
}

// ============================================================================
// Init
//
// This function is called from GameInfo and it initializes the log
// ============================================================================

function Init()
{
    if (!bTurnOff);
      TempLog = spawn(class 'FileLog');

    if (TempLog != None && !bTurnOff){
      TempLog.OpenLog(GetLogFilename(), "html");
    }
    else {
      Warn("Could not create output file");
    }
}


// ============================================================================
// NewGame
//
// New match has started
// ============================================================================

function NewGame()
{
  local string siServerName,
               siAdminName,
               siAdminEmail,
               siMOTD,
               ngTitle,
               ngAuthor,
               ngGameGameName;
  local int FragLimit,
            TimeLimit,
            MaxSpectators,
            NumBots,
            MaxPlayers,
            i,
            l;
  local bool bAllowBehindView,
             bAllowWeaponThrowing,
             bWeaponStay,
             bWeaponShouldViewShake;
  local float GameSpeed;

  local CTFFlag NewFlag;
  local Mutator Mut;

  foreach AllActors(class'Mutator',Mut)
  {
    if (Mut.IsA('MutUTComp')){
       Log("GameLogger: found UTComp");
       bUTCompIsPresent = True;
    }

    if (Mut.IsA('TTM_MutMain')){
       Log("GameLogger: found TTM");
       bTTMIsPresent = True;
    }
  }

   if (Level.Game.GameName == "Capture the Flag"){
    bCTF = True;
    foreach AllActors(class'CTFFlag', NewFlag)
    {
       if (NewFlag != None){
         l = FlagsArray.Length;
         FlagsArray.Length = l + 1;
         FlagsArray[l].thisFlag = NewFlag;
         FlagsArray[l].bLastSecond = False;
       }
    }
  }

  GRI = Level.Game.GameReplicationInfo;

  if (WSL!=None){
     WSL.NewGame();
  }

  if (IsTeamgame &&
			class'MutGameLogger'.default.MyBlueTeamName != "" &&
			class'MutGameLogger'.default.MyRedTeamName != ""){
    GRI.Teams[1].TeamName = class'MutGameLogger'.default.MyBlueTeamName;
    GRI.Teams[0].TeamName = class'MutGameLogger'.default.MyRedTeamName;
  }


  if (bDuelMode &&
			class'MutGameLogger'.default.MyBlueTeamName != "" &&
			class'MutGameLogger'.default.MyRedTeamName != ""){
    Player1 = class'MutGameLogger'.default.MyRedTeamName;
    Player2 = class'MutGameLogger'.default.MyBlueTeamName;
  }


  ngTitle                   = Level.Title;      // Making local copies
  ngAuthor                  = Level.Author;
  ngGameGameName            = Level.Game.default.GameName;
  FragLimit                 = Level.Game.GoalScore;
  TimeLimit                 = Level.Game.TimeLimit;
  GameSpeed                 = Level.Game.GameSpeed;
  MaxSpectators             = Level.Game.MaxSpectators;
  NumBots                   = Level.Game.NumBots;
  MaxPlayers                = Level.Game.MaxPlayers;
  bAllowBehindView          = Level.Game.bAllowBehindView;
  bAllowWeaponThrowing      = Level.Game.bAllowWeaponThrowing;
  bWeaponShouldViewShake    = Level.Game.bWeaponShouldViewShake;
  bWeaponStay               = Level.Game.bWeaponStay;
  siServerName              = GRI.ServerName; // Making local copies
  siAdminName               = GRI.AdminName;
  siAdminEmail              = GRI.AdminEmail;
  siMOTD                    = GRI.MessageOfTheDay;

  ReplaceText(siServerName,   tab, "_");
  ReplaceText(siAdminName,    tab, "_");
  ReplaceText(siAdminEmail,   tab, "_");
  ReplaceText(ngTitle,        tab, "_");  // Replacing tabs with _
  ReplaceText(ngAuthor,       tab, "_");
  ReplaceText(ngGameGameName, tab, "_");
  ReplaceText(siMOTD,         "|", "<br>");

  for (i = 0; i < ReportPlugins.Length; i++ ){
     ReportPlugins[i].NewGame(ngTitle,
                              ngAuthor,
                              ngGameGameName,
                              siServerName,
                              siAdminName,
                              siAdminEmail,
                              siMOTD);
  }

  myHTMLwrapper.WriteHeader( ngGameGameName, ngTitle);
  myHTMLwrapper.WriteMatchInfo(ngTitle,
                               ngAuthor,
                               ngGameGameName,
                               siServerName,
                               siAdminName,
                               siAdminEmail,
                               siMOTD );
}

//All the stuff is performed within NewGame();
function ServerInfo()
{
  if (WSL!=None){
    WSL.ServerInfo();
  }
}

// ============================================================================
//  StartGame
// Called when match starts
// ============================================================================
function StartGame()
{
  local int i;

	// I don't know how Epics deal with the StartGame being called twice
	// with UTComp - it's their own business, so I'll let the things as
	// they are w/out GameLogger.

  if (WSL!=None){
    WSL.StartGame();
  }

	// UTComp hack for ignoring warmup. they call it twice.
	if (bUTCompIsPresent && !bUTCompWarmUpStarted){
	  bUTCompWarmUpStarted = True;
	}
	else if (bUTCompIsPresent && bUTCompWarmUpStarted){
    bMatchStarted = True;
  }
  else if (!bUTCompIsPresent){
    bMatchStarted = True;
  }

	if (bMatchStarted){

		MatchStartTime = Level.TimeSeconds;

    for (i = 0; i < ReportPlugins.Length; i++ ){
      ReportPlugins[i].PreStartGame();
    }

  	myHTMLwrapper.StartGame(TimeStamp());

  	for (i = 0; i < ReportPlugins.Length; i++ ){
    	ReportPlugins[i].StartGame();
  	}
  }
}

// ============================================================================
//  ConnectEvent
// Called when someone connected to the server;
// ============================================================================
function ConnectEvent(PlayerReplicationInfo Who)
{
  local int i;

  for (i = 0; i < ReportPlugins.Length; i++ ){
    ReportPlugins[i].ConnectEvent(Who);
  }

  if (WSL!=None){
    WSL.ConnectEvent(Who);
  }

  if (bShowConnectsDisconnects && ((bIgnoreWarmupEvents && bMatchStarted) || !bIgnoreWarmupEvents)){
    if (bDebug){
      Logf("<!--Connect Event-->");
    }
    myHTMLwrapper.WriteSingleReportString(TimeStamp(), myHTMLwrapper.B(CCName(Who))@EnteredTheGameMsg);
  }
}

// ============================================================================
//  DisconnectEvent
// Called when someone disconnects from the server;
// ============================================================================
function DisconnectEvent(PlayerReplicationInfo Who)
{
  local int i;

  for (i = 0; i < ReportPlugins.Length; i++ ){
    ReportPlugins[i].DisconnectEvent(Who);
  }

  if (WSL!=None){
    WSL.DisconnectEvent(Who);
  }

  if (bShowConnectsDisconnects && ((bIgnoreWarmupEvents && bMatchStarted) || !bIgnoreWarmupEvents)){
    if (bDebug){
      Logf("<!--Disconnect Event-->");
    }
    myHTMLwrapper.WriteSingleReportString(TimeStamp(), myHTMLwrapper.B(CCName(Who))@LeftTheGameMsg);
  }
}

// ============================================================================
//  ScoreEvent
// Called when one of the players increases his/her score
// ============================================================================
function ScoreEvent(PlayerReplicationInfo Who, float Points, string Desc)
{
  local int i;

  // Calling the function in report plugins
  for (i = 0; i < ReportPlugins.Length; i++ ){
    ReportPlugins[i].ScoreEvent(Who, points, Desc);
  }

  // calling the function in world stats logger
  if (WSL!=None){
    WSL.ScoreEvent(who, points, desc);
  }

	if ( bIgnoreWarmupEvents && !bMatchStarted )
	  return;

  if (PlayerScoreEventsMode == 0)
    return;
  if (PlayerScoreEventsMode == 1 && ( Desc == "self_frag" || Desc == "tdm_frag" || Desc == "frag" || Desc == "team_frag" ))
    return;

  if (desc == "flag_denial"){
    LogLastSecondSave(Who.Team.TeamIndex);
  }

  if ( ClassifyEvent(Desc, "player") != "" ){
    if (bDebug){
      Logf("<!--Score Event-->");
    }
    myHTMLwrapper.WriteSingleReportString(TimeStamp(), myHTMLwrapper.B(CCName(Who))@PointsString(points)@ClassifyEvent(Desc, "player")@"("$int(Who.Score)$")");
  }
}

// ============================================================================
//  PickupEvent
// Called from the GameLogger.GameLoggerRules when one of the player picks up some
// item.
// ============================================================================

function PickupEvent(pawn Other, Pickup item)
{
  local string PlayerName;
  local string PHealthARmor;
  local string NewMessage;

  local bool bPickedWeapon;
  local bool bPickedHealth;
  local bool bPickedArmor;
  local bool bPickedUDamage;
  local bool bUseAT;

  local string Weapon;
  local string Armor;
  local string Health;
  local String PickUpLocation;
  local string UDamage;
  local string CAT;
  local string spanprefix;
  local string spansuffix;

  local int i;

	bUseAT=True;
  // Calling the function in report plugins
  for (i = 0; i < ReportPlugins.Length; i++ ){
     ReportPlugins[i].PickupEvent(Other, item);
  }

	if (bIgnoreWarmupEvents && !bMatchStarted)
	 return;

  PHealthARmor = "("$Other.Health$"H/"$string(int(Other.GetShieldStrength()))$"A)";
  PlayerName = Other.Controller.PlayerReplicationInfo.PlayerName;
  if (Other.Controller.PlayerReplicationInfo.PlayerZone != None)
    PickUpLocation = GL(Other.PlayerReplicationInfo);

  if (PickUpLocation == ""){
	  bUseAT=false;
		PickUpLocation = " ";
	}
	if (Left(PickUpLocation, 5) ~= "Near ")
	  bUseAT=false;
	/*
	if (bDebug){
	  Log("Picked up item location:"@PickUpLocation);
	  Log("UseAT is"@bUseAT);
	}
	*/
  // Weapon PickUps
  //================================================
  // There are no ShieldGun Pickups

  if (item.IsA('AssaultRiflePickup')){
    Weapon = PickedWeaponNames[1];
    bPickedWeapon=true;
  }
  //yes I know, that there are no AR pickups at standard maps, but they
  //can be at custom ones
  else if (item.IsA('BioRiflePickup')){
    Weapon = PickedWeaponNames[2];
    bPickedWeapon=true;
  }
  else if (item.IsA('ShockRiflePickup')){
    Weapon = PickedWeaponNames[3];
    bPickedWeapon=true;
  }
  else if (item.IsA('LinkGunPickup')){
    Weapon = PickedWeaponNames[4];
    bPickedWeapon=true;
  }
  else if (item.IsA('MinigunPickup')){
    Weapon = PickedWeaponNames[5];
    bPickedWeapon=true;
  }
  else if (item.IsA('FlakCannonPickup')){
    Weapon = PickedWeaponNames[6];
    bPickedWeapon=true;
  }
  else if (item.IsA('RocketLauncherPickup')){
    Weapon = PickedWeaponNames[7];
    bPickedWeapon=true;
  }
  else if (item.IsA('SniperRiflePickup')){
    Weapon = PickedWeaponNames[8];
    bPickedWeapon=true;
  }
  else if (item.IsA('ClassicSniperRiflePickup')){
    Weapon = PickedWeaponNames[9];
    bPickedWeapon=true;
  }
  else if (item.IsA('ONSAVRiLPickup')){
    Weapon = PickedWeaponNames[10];
    bPickedWeapon=true;
  }
  else if (item.IsA('ONSGrenadePickup')){
    Weapon = PickedWeaponNames[11];
    bPickedWeapon=true;
  }
  else if (item.IsA('ONSMineLayerPickup')){
    Weapon = PickedWeaponNames[12];
    bPickedWeapon=true;
  }
  // There are no Translocator Pickups too =)
  else if (item.IsA('RedeemerPickup')){
    Weapon = myHTMLwrapper.B(PickedWeaponNames[14]);
    bPickedWeapon=true;
  }
  else if (item.IsA('ONSPainterPickup')){
    Weapon = PickedWeaponNames[17];
    bPickedWeapon=true;
  }
  else if (item.IsA('PainterPickup')){
    Weapon = PickedWeaponNames[15];
    bPickedWeapon=true;
  }
  else if (item.IsA('SuperShockRiflePickup')){
    Weapon = PickedWeaponNames[16];
    bPickedWeapon=true;
  }
  // Armor PickUps
  //================================================
  else if (item.IsA('ShieldPack')){
    Armor = ArmorNames[0];
    bPickedArmor=true;
  }
  else if (item.IsA('SuperShieldPack')){
    Armor = myHTMLwrapper.B(ArmorNames[1]);
    bPickedArmor=true;
  }
  // Health PickUps
  //================================================
  //don't log vials, they'll cause to much spam;
  else if (item.IsA('HealthPack')){
    Health = HealthNames[0];
    bPickedHealth=true;
  }
  else if (item.IsA('SuperHealthPack')){
    Health = myHTMLwrapper.B(HealthNames[1]);
    bPickedHealth=true;
  }
  // UDamage PickUp
  //================================================
  else if (item.IsA('UDamagePack')){
    UDamage = myHTMLwrapper.B(UDamageName);
    bPickedUDamage=true;
  }

  //localization stuff
  if (bUseAT){
    CAT = ATsuffix;
  }
  else {
    CAT = "";
  }

  if (bPickedWeapon){
    newMessage = PlayerName$PHealthARmor$picks$Weapon@CAT$PickUpLocation;
  }

  if (bPickedArmor){
    newMessage = PlayerName$PHealthARmor$picks$Armor@CAT$PickUpLocation;
  }

  if (bPickedHealth){
    newMessage = PlayerName$PHealthARmor$picks$Health@CAT$PickUpLocation;
  }

  if (bPickedUDamage){
    newMessage = PlayerName$PHealthARmor$picks$UDamage@CAT$PickUpLocation;
  }

  // this event is called several times, so to prevent it we check if
  // the previosly logged string is looking the same. if so, return;
  if (NewMessage ~= TempPickupString || NewMessage == "" )
    return;

  if(bPickedWeapon && !bLogWeaponsPicks)
    return;

  if( ( bPickedArmor || bPickedHealth || bPickedUDamage ) && !bLogOtherPicks)
    return;

  if (bDebug){
    Logf("<!--Pickup Event-->");
  }

  if (bColoredMessages && ( bPickedArmor || bPickedHealth || bPickedUDamage )){
    spanprefix = "<span class=item_pickup>";
    spansuffix = "</span>";
  }
  if (bColoredMessages && bPickedWeapon){
    spanprefix = "<span class=weapon_pickup>";
    spansuffix = "</span>";
  }

  myHTMLwrapper.WriteSingleReportString(Timestamp(), spanprefix$newMessage$spansuffix);

  TempPickupString = NewMessage;// Saving the new string, to check for pickup spamming
}

// ============================================================================
//  TeamScoreEvent
// Called when one of the teams increases its score
// ============================================================================
function TeamScoreEvent(int Team, float Points, string Desc)
{
  local int i;
  local string TSEpoints;
  local string TSEEvent;
  local string ThisTeamName;

	if (bIgnoreWarmupEvents && !bMatchStarted)
	 return;

  ThisTeamName = CTName(team);

  TSEpoints = PointsString(points, Desc);
  TSEEvent  = ClassifyEvent(Desc, "team");

  if (WSL!=None){
    WSL.TeamScoreEvent(team, points, desc);
  }

  for (i = 0; i < ReportPlugins.Length; i++){
     ReportPlugins[i].TeamScoreEvent(team, points, desc);
  }

  if (bShowTeamScoreEvents && Desc != "tdm_frag"){ // this is to prevent spam in TDM;
    if (bDebug){
      Logf("<!--Team Score Event-->");
    }
    myHTMLwrapper.WriteSingleReportString(Timestamp(), myHTMLwrapper.B(ThisTeamName)@TSEpoints@TSEevent);
  }

  if (Desc ~= "enemy_core_destroyed"){
    if (team == 1){
      myHTMLwrapper.WriteSingleReportString(Timestamp(), myHTMLwrapper.B(CTName(0))@ONSCoreDestroyed);
    }
    else{
     myHTMLwrapper.WriteSingleReportString(Timestamp(), myHTMLwrapper.B(CTName(1))@ONSCoreDestroyed);
    }
    // No "New round started" if one of teams reaches or exceeds goal score
    if ((Level.Game.GameReplicationInfo.Teams[team].Score + Points) < Level.Game.GoalScore && Level.Game.GoalScore > 0)
      myHTMLwrapper.WriteColspanTwoString(NewRoundStarted2);
  }
}

// ============================================================================
//  HDamageEvent
// Called from GameLogger.GameLoggerRules when one of the players inflicts to
// another damage higher than defined damage threshold with non-instant kill
// damagetype;
// Function isn't called at all, if self.HeavyDamageReportMode is equal to 0;
// ============================================================================
function HDamageEvent(pawn ImHit, pawn ByMe, int Damage, class<DamageType> DamageType)
{
  //Local Vars
  local string NewString; //the result string
  local string victimName, shooterName;
  local string victimLocation; //location of the killed player

  local int i;

  // grammar control variables
  local bool bHeadshot; //is this kill is a Lightning Gun headshot?
  local bool bClassicHeadshot; //Is this kill is a sniper Rifle headshot?
  local bool bAltWith; //special for russian language (and others alike) translation
  local bool bUseAT;

  local string lNATPrefix;
  local string MyDamage;
  local string selfDamagestring;
  local string leavingonly;
  local string DamagePercentBy;
  local string CWith;
  local string CAT;
  local string SpanPrefix;
  local string SpanSuffix;

  bUseAt = true;

  for (i = 0; i < ReportPlugins.Length; i++){
     ReportPlugins[i].HDamageEvent(ImHit, ByMe, Damage, DamageType);
  }

 	if ( bIgnoreWarmupEvents && !bMatchStarted )
	  return;

  if (ImHit == None || ImHit.Controller == None || ImHit.Controller.PlayerReplicationInfo == None)
    return;


  if (ByMe == None || ByMe.Controller == None || ByMe.Controller.PlayerReplicationInfo == None)
    return;   //whatever dealt damage it's not one of the match participats

  //Location stuff
  victimLocation = GL(imHit.Controller.PlayerReplicationInfo);

  if (victimLocation == ""){
    bUseAT=false;
		victimLocation = " ";
  }
 	if (Left(victimLocation, 5) ~= "Near ")
	  bUseAT=false;
	/*
	if (bDebug){
	  Log("Heavy damage event happened"@victimLocation);
	  Log("UseAT is"@bUseAT);
	}
	*/
  bAltWith   = false;
  lNATprefix = NATprefix;

  victimName  = myHTMLwrapper.B(CCName(imHit.Controller.PlayerReplicationInfo));
  shooterName = myHTMLwrapper.B(CCName(Byme.Controller.PlayerReplicationInfo));

  MyDamage = "None";

  //--------------------------------------------
  //  Weapon caused deaths
  //--------------------------------------------
  if(damagetype == class'DamTypeAssaultBullet'){
    MyDamage = WeaponNames[1]$firetype[0];
  }
  else if(damagetype == class'DamTypeAssaultGrenade'){
    MyDamage = WeaponNames[1]$firetype[1];
  }
  else if(damagetype == class'DamTypeBioGlob'){
    MyDamage = WeaponNames[2];
    bAltWith = true;
  }
  else if(damagetype == class'DamTypeFlakChunk'){
    MyDamage = WeaponNames[6]$firetype[0];
  }
  else if(damagetype == class'DamTypeFlakShell'){
    MyDamage = WeaponNames[6]$firetype[1];
  }
  else if(damagetype == class'DamTypeLinkPlasma'){
    MyDamage = WeaponNames[4]$firetype[0];
  }
  else if(damagetype == class'DamTypeLinkShaft'){
    MyDamage = WeaponNames[4]$firetype[1];
  }
  else if(damagetype == class'DamTypeMinigunAlt'){
    MyDamage = WeaponNames[5]$firetype[1];
  }
  else if(damagetype == class'DamTypeMinigunBullet'){
    MyDamage = WeaponNames[5]$firetype[0];
  }
  else if(damagetype == class'DamTypeRedeemer'){
    MyDamage = WeaponNames[14];
    bAltWith = true;
  }
  else if(damagetype == class'DamTypeRocket'){
    MyDamage = WeaponNames[7];
    bAltWith = true;
  }
  else if(damagetype == class'DamTypeRocketHoming'){
    MyDamage = WeaponNames[7];
    bAltWith = true;
  }
  else if(damagetype == class'DamTypeShieldImpact'){
    MyDamage = WeaponNames[0];
    bAltWith = true;
  }
  else if(damagetype == class'DamTypeShockBall'){
    MyDamage = WeaponNames[3]$firetype[1];
  }
  else if(damagetype == class'DamTypeShockBeam'){
    MyDamage = WeaponNames[3]$firetype[0];
  }
  else if(damagetype == class'DamTypeClassicSniper'){
    MyDamage = WeaponNames[9];
    bAltWith = true;
  }
  else if(damagetype == class'DamTypeSniperShot'){
    MyDamage = WeaponNames[8];
    bAltWith = true;
  }
  else if(damagetype == class'DamTypeSuperShockBeam'){
    MyDamage = WeaponNames[16];
    bAltWith = true;
  }
  else if(damagetype == class'DamTypeONSAVRiLRocket'){
    MyDamage = WeaponNames[10];
    bAltWith = true;
  }
  else if(damagetype == class'DamTypeONSGrenade'){
    MyDamage = WeaponNames[11];
    bAltWith = true;
  }
  else if(damagetype == class'DamTypeONSMine'){
    MyDamage = WeaponNames[12];
    bAltWith = true;
  }

  //-------------------------------------------------
  //  Weapon caused special deaths
  //-------------------------------------------------
  else if(damagetype == class'DamTypeClassicHeadshot'){
    MyDamage = WeaponNames[9];
    bClassicHeadshot = true;
  }
  else if(damagetype == class'DamTypeSniperHeadShot'){
    MyDamage = WeaponNames[8];
    bHeadshot = true;
  }
  else if(damagetype == class'DamTypeShockCombo'){
    MyDamage = Combo;
  }

  //-------------------------------------------------
  //  Onslaught vehicle's weapons damage types
  //-------------------------------------------------
  else if(damagetype == class'DamTypeAttackCraftMissle'){
    MyDamage = RaptorMissile;
  }
  else if(damagetype == class'DamTypeAttackCraftPlasma'){
    MyDamage = RaptorPlasma;
  }
  else if(damagetype == class'DamTypeBallTurretPlasma'){
    MyDamage = HBNDSideTurretSec;
  }
  else if(damagetype == class'DamTypeChargingBeam'){
    MyDamage = HBNDBackTurret;
  }
  else if(damagetype == class'DamTypeHoverBikePlasma'){
    MyDamage = MantaPlasma;
  }
  else if(damagetype == class'DamTypeLinkTurretPlasma'){
    MyDamage = LinkTurretPlasma;
  }
  else if(damagetype == class'DamTypeLinkTurretBeam'){
    MyDamage = LinkTurretBeam;
  }
  else if(damagetype == class'DamTypeMASPlasma'){
    MyDamage = LeviTurret;
  }
  else if(damagetype == class'DamTypeMinigunTurretBullet'){
    MyDamage = MiniTurret;
  }
  else if(damagetype == class'DamTypeONSChainGun'){
    MyDamage = GoliathTurret;
  }
  else if(damagetype == class'DamTypePRVCombo'){
    MyDamage = HBNDSideTurretCombo;
  }
  else if(damagetype == class'DamTypePRVLaser'){
    MyDamage = HBNDSideTurretPri;
  }
  else if(damagetype == class'DamTypeSpaceFighterLaser'){
    MyDamage = HFighterLaser;
  }
  else if(damagetype == class'DamTypeSpaceFighterLaser_Skaarj'){
    MyDamage = SFighterLaser;
  }
  else if(damagetype == class'DamTypeSpaceFighterMissile'){
    MyDamage = HFighterMissile;
  }
  else if(damagetype == class'DamTypeSpaceFighterMissileSkaarj'){
    MyDamage = SFighterMissile;
  }
  else if(damagetype == class'DamTypeTurretBeam'){
    MyDamage = ShockTurret;
  }
  else{
    MyDamage = CheckForDamageInPlugins(damagetype);
  }

  //-----------------------------------------------------
  // Default return
  //-----------------------------------------------------
  if (MyDamage ~= "None") {
    MyDamage = WeaponNames[17];
  }

  //translation support voodoo
  if (bAltWith){
    CWith = AWith;
  }
  else {
    CWith = With;
  }

  if (bUseAT){
    CAT = ATsuffix;
  } else {
    CAT = "";
  }

  DamagePercentBy = string(Damage)$damagestring@bystring;

  if (ImHit.Controller.PlayerReplicationInfo.bIsFemale)
    selfDamagestring = heavyselfdamageF;
  else
    selfDamagestring = heavyselfdamageM;

  if (bHeadshot || bClassicHeadshot){
    if(ImHit.Controller.PlayerReplicationInfo.bIsFemale)
      leavingonly = leavingher;
    else
      leavingonly = leavinghim;
  }

  if (bColoredMessages){
    SpanPrefix = "<span class=\"HdamageEvent\">";
    SpanSuffix = "</span>";
  }

  if ( ImHit == ByMe )//teh suicide attempt
  {
    NewString = victimName$SpanPrefix@selfDamagestring@DamagePercentBy@MyDamage@CAT$victimLocation$SpanSuffix;
  }
  else
  {
    if (bHeadshot){
      NewString = shooterName$SpanPrefix$Headshot$SpanSuffix$victimname$SpanPrefix$headsuffix@CAT$victimLocation@Cwith$MyDamage@leavingonly@string(ImHit.Health - Damage)$"%"$SpanSuffix;
    }
    else if (bClassicHeadshot){
      NewString = shootername$SpanPrefix$Headshot$SpanSuffix$victimname$SpanPrefix$headsuffix@CAT$victimLocation@Cwith$MyDamage@leavingonly@string(ImHit.Health - Damage)$"%"$SpanSuffix;
    }
    else {
      NewString = shootername$SpanPrefix$damagesstring$SpanSuffix$victimname$SpanPrefix@CAT$victimLocation@bystring@string(Damage)$"%"@Cwith$MyDamage$SpanSuffix;
    }
  }

  if (bDebug){
    Logf("<!--HDamage Event-->");
  }
  myHTMLwrapper.WriteSingleReportString(Timestamp(), NewString);
}

// ==============================================================
// NodeStateChange
//
// being called crom Timer() when it founds that one of the nodes
// has changed it's state
// ==============================================================

function NodeStateChange( ONSPowerNodeNeutral Node, optional byte PrevState)
{
  local byte Stage;
  local int OwnerTeamIndex, DestroyerTeamIndex;
  local string Result;

	if ( bIgnoreWarmupEvents && !bMatchStarted )
	  return;

  Stage = Node.CoreStage;

  OwnerTeamindex = Node.Constructor.PlayerReplicationInfo.Team.TeamIndex;
  if (Node.Constructor.PlayerReplicationInfo.Team.TeamIndex == 1){
    DestroyerTeamIndex = 0;
  }
  else{
    DestroyerTeamIndex = 1;
  }

  if (Stage == 0){
    result = ONSPowerNodeName@Node.NodeNum@ONSPowerNode_constructed@CTName(OwnerTeamindex);
  }
  else if (Stage == 1){
    if (!Node.bSevered)
      result = ONSPowerNodeName@Node.NodeNum@ONSPowerNode_destroyed@CTName(DestroyerTeamIndex);
    else result = ONSPowerNodeName@Node.NodeNum@ONSPowerNode_destroyeddueisolation;
  }
  else if (Stage == 2){
    result = ONSPowerNodeName@Node.NodeNum@ONSPowerNode_beingconstructed@CTName(OwnerTeamindex);
  }
  else if (Stage == 255){
    result = ONSPowerNodeName@Node.NodeNum@ONSPowerNode_disabled;
  }
  else{
    return;
  }

  if (bDebug){
    Logf("<!--Node Change State Event-->");
  }
  myHTMLwrapper.WriteSingleReportString(TimeStamp(), myHTMLwrapper.B(result));
}

// ===========================================================================
//  Someone died...
// ===========================================================================
function KillEvent(string Killtype, PlayerReplicationInfo Killer, PlayerReplicationInfo Victim, class<DamageType> Damage)
{
  //Local Vars
  local string NewString; //the result string
  local string KillerName, KilledName;
  local string KilledLocation; //location of the killed player

  local int i;

  // grammar control variables
  local bool bHeadshot; //is this kill is a Lightning Gun headshot?
  local bool bClassicHeadshot; //Is this kill is a sniper Rifle headshot?
  local bool bNatural; //Is this a natural death (i.e. lava swim)?
  local bool bCrush, bPush; //is player crushed or was pushed somewhere
  local bool bWasTyping; //victim was in console while got killed
  local bool bPancake; //Pancake
  local bool bRanOver;
  local bool bClippedHeadOff;// Manta
  local bool bAltWith; //special for russian language (and others alike) translation
  local bool bUseAT;

  local string lCrush;
  local string lPush;
  local string lNATPrefix;
  local string lWhileTyping;
  local string lpushsuffix;
  local string MyDamage;
  local string SDamage; //(Suicide for natural kills)
  local string CWith;
  local string CAT;

	bUseAt 						= True;
	bPush 						= false;
	bCrush 						= false;
	bRanOver 					= false;
	bClippedHeadOff 	= false;
	bNatural 					= false;
	bClassicHeadshot  = false;
	bHeadshot					= false;


  //WSL call
  if (WSL!=None){
    WSL.KillEvent(killtype, killer, victim, damage);
  }

  for (i = 0; i < ReportPlugins.Length; i++){
    ReportPlugins[i].KillEvent(killtype, killer, victim, damage);
  }

	if ( bIgnoreWarmupEvents && !bMatchStarted )
	  return;

  //Location stuff
  KilledLocation = GL(Victim);

  if (KilledLocation == ""){
    bUseAT=false;
		KilledLocation = " ";
  }

	if (Left(KilledLocation, 5) ~= "Near ")
	  bUseAT=false;
	/*
	if (bDebug){
		Log("Victim location when killed is"@KilledLocation);
	  Log("UseAT is"@bUseAT);
  }
  */
  bAltWith=false;

  KilledName = myHTMLwrapper.B(CCName(Victim));
  KillerName = myHTMLwrapper.B(CCName(Killer));

  lNATprefix  = NATprefix;
  lCrush      = crush;
  lPush       = push;
  lpushsuffix = pushsuffix;


  if (Victim.bIsFemale){
    lWhileTyping = WhileTypingF;
  }
  else
    lWhileTyping = WhileTypingM;

  // it seems bots are always "in console", according to my test runs.
//  if (Victim.bBot)
//    lWhileTyping="";

  if ( PlayerController(Victim.Owner)!= None && PlayerController(Victim.Owner).bIsTyping)
  {
	if ( PlayerController(Killer.Owner) != PlayerController(Victim.Owner) )
//	  SpecialEvent(Killer, "type_kill");						// Killer killed typing victim
	  bWasTyping = True;
  }

  MyDamage = "None";

//--------------------------------------------
//  Weapon caused deaths
//--------------------------------------------
  if(damage == class'DamTypeAssaultBullet'){
    MyDamage = WeaponNames[1]$firetype[0];
  }
  else if(damage == class'DamTypeAssaultGrenade'){
    MyDamage = WeaponNames[1]$firetype[1];
  }
  else if(damage == class'DamTypeBioGlob'){
    MyDamage = WeaponNames[2];
    bAltWith=true;
  }
  else if(damage == class'DamTypeFlakChunk'){
    MyDamage = WeaponNames[6]$firetype[0];
  }
  else if(damage == class'DamTypeFlakShell'){
    MyDamage = WeaponNames[6]$firetype[1];
  }
  else if(damage == class'DamTypeIonBlast'){
    MyDamage = WeaponNames[15];
    bAltWith=true;
  }
  else if(damage == class'DamTypeLinkPlasma'){
    MyDamage = WeaponNames[4]$firetype[0];
  }
  else if(damage == class'DamTypeLinkShaft'){
    MyDamage = WeaponNames[4]$firetype[1];
  }
  else if(damage == class'DamTypeMinigunAlt'){
    MyDamage = WeaponNames[5]$firetype[1];
  }
  else if(damage == class'DamTypeMinigunBullet'){
    MyDamage = WeaponNames[5]$firetype[0];
  }
  else if(damage == class'DamTypeRedeemer'){
    MyDamage = WeaponNames[14];
    bAltWith=true;
  }
  else if(damage == class'DamTypeRocket'){
    MyDamage = WeaponNames[7];
    bAltWith=true;
  }
  else if(damage == class'DamTypeRocketHoming'){
    MyDamage = WeaponNames[7];
    bAltWith=true;
  }
  else if(damage == class'DamTypeShieldImpact'){
    MyDamage = WeaponNames[0];
    bAltWith=true;
  }
  else if(damage == class'DamTypeShockBall'){
    MyDamage = WeaponNames[3]$firetype[1];
  }
  else if(damage == class'DamTypeShockBeam'){
    MyDamage = WeaponNames[3]$firetype[0];
  }
  else if(damage == class'DamTypeClassicSniper'){
    MyDamage = WeaponNames[9];
    bAltWith=true;
  }
  else if(damage == class'DamTypeSniperShot'){
    MyDamage = WeaponNames[8];
    bAltWith=true;
  }
  else if(damage == class'DamTypeSuperShockBeam'){
    MyDamage = WeaponNames[16];
    bAltWith=true;
  }
  else if(damage == class'DamTypeONSAVRiLRocket'){
    MyDamage = WeaponNames[10];
    bAltWith=true;
  }
  else if(damage == class'DamTypeONSGrenade'){
    MyDamage = WeaponNames[11];
    bAltWith=true;
  }
  else if(damage == class'DamTypeONSMine'){
    MyDamage = WeaponNames[12];
    bAltWith=true;
  }
  else if(damage == class'DamTypeTeleFrag'){
    MyDamage = WeaponNames[13];
    bAltWith=true;
  }
  // -------------------------------------------------
  // Weapon caused special deaths
  // -------------------------------------------------
  else if(damage == class'DamTypeClassicHeadshot'){
    MyDamage = WeaponNames[9];
    bClassicHeadshot=true;
  }
  else if(damage == class'DamTypeSniperHeadShot'){
    MyDamage = WeaponNames[8];
    bHeadshot=true;
  }
  else if(damage == class'DamTypeShockCombo'){
    MyDamage = Combo;
  }
  // -------------------------------------------------
  // Onslaught vehicle's weapons damage types
  // -------------------------------------------------
  else if(damage == class'DamTypeAttackCraftMissle'){
    MyDamage = RaptorMissile;
  }
  else if(damage == class'DamTypeAttackCraftPlasma'){
    MyDamage = RaptorPlasma;
  }
  else if(damage == class'DamTypeBallTurretPlasma'){
    MyDamage = HBNDSideTurretSec;
  }
  else if(damage == class'DamTypeChargingBeam'){
    MyDamage = HBNDBackTurret;
  }
  else if(damage == class'DamTypeHoverBikePlasma'){
    MyDamage = MantaPlasma;
  }
  else if(damage == class'DamTypeLinkTurretPlasma'){
    MyDamage = LinkTurretPlasma;
  }
  else if(damage == class'DamTypeLinkTurretBeam'){
    MyDamage = LinkTurretBeam;
  }
  else if(damage == class'DamTypeIonTankBlast'){
    MyDamage = IonTankCannon;
  }
  else if(damage == class'DamTypeIonCannonBlast'){
    MyDamage = IonCannonShot;
  }
  else if(damage == class'DamTypeMASCannon'){
    MyDamage = LeviBlast;
  }
  else if(damage == class'DamTypeMASPlasma'){
    MyDamage = LeviTurret;
  }
  else if(damage == class'DamTypeMinigunTurretBullet'){
    MyDamage = MiniTurret;
  }
  else if(damage == class'DamTypeONSChainGun'){
    MyDamage = GoliathTurret;
  }
  else if(damage == class'DamTypeONSRVBlade'){
    MyDamage = ScorpBlade;
    bClippedHeadOff=true;
  }
  else if(damage == class'DamTypeONSWeb'){
    MyDamage = ScorpNet;
  }
  else if(damage == class'DamTypePRVCombo'){
    MyDamage = HBNDSideTurretCombo;
  }
  else if(damage == class'DamTypePRVLaser'){
    MyDamage = HBNDSideTurretPri;
  }
  else if(damage == class'DamTypeSpaceFighterLaser_Skaarj'){
    MyDamage = SFighterLaser;
  }
  else if(damage == class'DamTypeSpaceFighterLaser'){
    MyDamage = HFighterLaser;
  }
  else if(damage == class'DamTypeSpaceFighterMissileSkaarj'){
    MyDamage = SFighterMissile;
  }
  else if(damage == class'DamTypeSpaceFighterMissile'){
    MyDamage = HFighterMissile;
  }
  else if(damage == class'DamTypeTankShell'){
    MyDamage = GoliathCannon;
  }
  else if(damage == class'DamTypeTurretBeam'){
    MyDamage = ShockTurret;
  }
  // -------------------------------------------------
  // Vehicle caused "natural" deaths (roadkills)
  // -------------------------------------------------
  else if(damage == class'DamTypeAttackCraftRoadkill'){
    MyDamage = RaptorRK;
    bRanOver=True;
  }
  else if(damage == class'DamTypeHoverBikeHeadshot'){
    MyDamage = MantaHeadClip;
    bClippedHeadOff=True;
  }
  else if(damage == class'DamTypeIonTankRoadkill'){
    MyDamage = IonTankRK;
    bRanOver=True;
  }
  else if(damage == class'DamTypeMASRoadkill'){
    MyDamage = LeviRK;
    bRanOver=True;
  }
  else if(damage == class'DamTypeAttackCraftPancake'){
    MyDamage = RaptorPK;
    bPancake=True;
  }
  else if(damage == class'DamTypeHoverBikePancake'){
    MyDamage = MantaPK;
    bPancake=True;
  }
  else if(damage == class'DamTypeIonTankPancake'){
    MyDamage = IonTankPK;
    bPancake=True;
  }
  else if(damage == class'DamTypeMASPancake'){
    MyDamage = LeviPK;
    bPancake=True;
  }
  else if(damage == class'DamTypePRVPancake'){
    MyDamage = HBND_PK;
    bPancake=True;
  }
  else if(damage == class'DamTypeRVPancake'){
    MyDamage = ScorpPK;
    bPancake=True;
  }
  else if(damage == class'DamTypeTankPancake'){
    MyDamage = GoliathPK;
    bPancake=True;
  }
  else if(damage == class'DamTypePRVRoadkill'){
    MyDamage = HBND_RK;
    bRanOver=True;
  }
  else if(damage == class'DamTypeRVRoadkill'){
    MyDamage = ScorpRK;
    bRanOver=True;
  }
  else if(damage == class'DamTypeDestroyedVehicleRoadKill'){
    MyDamage = BlownVehicleRK;
    bCrush=True;
  }
  else if(damage == class'DamTypeTankRoadkill'){
    MyDamage = GoliathRK;
    bRanOver=True;
  }
  else if(damage == class'DamTypeONSVehicleExplosion'){
    MyDamage = VehicleExplosion;
  }
  else if(damage == class'DamTypeONSVehicle'){
    MyDamage = Accident;
  }
  // -------------------------------------------------
  // Natural Deaths
  // -------------------------------------------------
  else if(damage == class'Burned'){
    SDamage  = SBurned;
    MyDamage = Burned;
    bNatural =true;
  }
  else if(damage == class'Corroded'){
    SDamage  = SCorroded;
    MyDamage = Corroded;
    bNatural = true;
  }
  else if(damage == class'Crushed'){
    SDamage  = SCrushed;
    MyDamage = Crushed;
    bNatural = true;
    bCrush   = true;
  }
  else if(damage == class'Drowned'){
    SDamage  = SDrowned;
    MyDamage = Drowned;
    bNatural = true;
  }
  else if(damage == class'fell'){
    SDamage  = Sfell;
    MyDamage = fell;
    bNatural = true;
    bPush    = True;
  }
  else if(damage == class'Burned'){
    SDamage  = Sfelllava;
    MyDamage = felllava;
    bNatural = true;
    bPush    = true;
  }
  else{
    MyDamage = CheckForDamageInPlugins(damage);
  }

  // -----------------------------------------------------
  // Default return
  // -----------------------------------------------------
  if (MyDamage ~= "None") {
    MyDamage = WeaponNames[17];
  }
   // translation support voodoo

   // to avoid deleting defaults

    if (bAltWith){
      CWith = AWith;
    }
    else {
      CWith = With;
    }

    if (bUseAT){
      CAT = ATsuffix;
    }
    else {
      CAT = "";
    }

    if (bPush && Killer != Victim && Killer != None){
      Log("bPush is true and Victim != Killer");
      lNATprefix = "";
      lCrush     = "";
      MyDamage   = "";
    }
    else if (bPush && ( (Killer == Victim) || Killer == None)){
      lNATprefix = "";
      lCrush     = "";
    }
    else if (bCrush){
      lNATprefix  = "";
      lPush       = "";
      lpushsuffix = "";
    }
    else if (!bCrush && !bPush){
      lCrush      = "";
      lpushsuffix = "";
      lPush       = "";
    }

    if (!bWasTyping){
      lWhileTyping = "";
    }

    if ( Killer == Victim )
    {
       if (!bNatural)
         NewString = KilledName$suicidestring$Cwith$MyDamage@CAT$KilledLocation;
       else
         NewString = KilledName$SDamage@CAT$KilledLocation;
    }
    else if (Killer == None){ // usually an ONSVehcile Damage, aka crashed into smth and died.
       if (!bNatural)
         NewString = KilledName$accident@CAT$KilledLocation;
       else if (bNatural && bPush)
         NewString = KilledName$SDamage@CAT$KilledLocation;
       else
         NewString = KilledName$MyDamage@CAT$KilledLocation;
    }
    else
    {
       if (bHeadshot){
         NewString = KillerName$Headshot$KilledName$headsuffix@CAT$KilledLocation@Cwith$MyDamage$lwhiletyping;
       }
       else if (bClassicHeadshot){
         NewString = KillerName$Headshot$KilledName$headsuffix@CAT$KilledLocation@Cwith$MyDamage$lwhiletyping;
       }
       else if (bNatural){
         NewString = KillerName$lcrush$lpush$lNATprefix$KilledName$lpushsuffix@CAT$KilledLocation@MyDamage$lwhiletyping;
       }
       else if (bPanCake){
         NewString = KillerName$lands_on$KilledName@CAT$KilledLocation$MyDamage@lwhiletyping;
       }
       else if (bRanOver){
         NewString = KillerName$runsover$KilledName@CAT$KilledLocation$MyDamage@lwhiletyping;
       }
       else if (bClippedHeadOff){
         NewString = KillerName$clips$KilledName$MyDamage@CAT$KilledLocation@lwhiletyping;
       }
       else{
         NewString = KillerName$killsstring$KilledName@CAT$KilledLocation$Cwith$MyDamage@lwhiletyping;
       }
    }
    if (bDebug) {
		  Logf("<!--Kill Event-->");
		  Logf("<!-- Don't Log kills:"@bDontLogKills$", bPanCake is"@bPanCake$", bNatural is"@bNatural$", bClassicHeadshot is"@bClassicHeadshot$", bheadshot is"@bheadshot$". -->");
		  Log("bPush is "$bPush$" bCrush is "$bCrush$" Killer is "$KillerName$" and victim is "$KilledName);
		  LogF("<!-- Damagetype is"@Damage$" -->");
		}

		if (!bDontLogKills || (bDontLogKills && (bPanCake || bNatural || bClassicHeadshot || bHeadshot || Killer == Victim || bWasTyping)))
      myHTMLwrapper.WriteSingleReportString(Timestamp(), NewString);
}

// ==========================================================================
//  Game has ended;
// ==========================================================================
function EndGame(string Reason)
{
  local int i,j;
  local array<PlayerReplicationInfo> PRIs;
  local PlayerReplicationInfo PRI,t;

  if (WSL!=None){
    WSL.EndGame(reason);
  }

  // Quick cascade sort.
  for (i=0;i<GRI.PRIArray.Length;i++)
  {
    PRI = GRI.PRIArray[i];

    PRIs.Length = PRIs.Length+1;
    for (j=0;j<Pris.Length-1;j++)
    {
      if (PRIs[j].Score < PRI.Score ||
           (PRIs[j].Score == PRI.Score && PRIs[j].Deaths > PRI.Deaths)
      )
      {
        t = PRIs[j];
        PRIs[j] = PRI;
        PRI = t;
      }
    }
    PRIs[j] = PRI;
  }

  // sorting off OnlySpectators, among these are WebAdmins, DemoRecSpectators etc.
  if (bSortOffOnlySpecs){
    for (i = 0; i < PRIs.Length; i++){
      if (PRIs[i].bOnlySpectator){
        PRIs.Remove(i, 1);
        i = i - 1;
      }
    }
  }

  for (i = 0; i < ReportPlugins.Length; i++ ){
     ReportPlugins[i].BeforeReportEnd(PRIs, GRI, reason);
  }

  myHTMLwrapper.WriteSingleReportString(TimeStamp(), GameEndedMsg);
  myHTMLwrapper.WriteReportTableEnd();

  bClosedReport = True;

  for (i = 0; i < ReportPlugins.Length; i++ ){
     ReportPlugins[i].AfterReportEnd(PRIs, GRI, reason);
  }

  myHTMLwrapper.WriteScoreBoardheader();
  myHTMLwrapper.WriteScoreboard(PRIs, GRI);
  myHTMLwrapper.WriteScoreboardBottom();

  for (i = 0; i < ReportPlugins.Length; i++ ){
     ReportPlugins[i].AfterScoreboard(PRIs, GRI, reason);
  }

  if (bExportDetailedStats){
    myHTMLwrapper.WriteDetailedPlayerStats(PRIs);
  }

  if (IsTeamGame){
    if (GRI.Teams[0].Score > GRI.Teams[1].Score)
      Winner = GRI.Teams[0].TeamName;
    else
      Winner = GRI.Teams[1].TeamName;
  }
  else{
    if (PRIs[0].PlayerName != "WebAdmin" || PRIs[0].PlayerName != "DemoRecSpectator" || !PRIs[0].bIsSpectator || !PRIs[0].bOnlySpectator )
      Winner = PRIs[0].PlayerName;
    else winner = "Nobody";
  }
  myHTMLwrapper.WriteDocumentEnd(Winner);
}

// ============================================================================
// GameEvent
//
// Logs game events like name changes, of flag/bomb being fropped or carried
// ============================================================================
function GameEvent(string GEvent, string Desc, PlayerReplicationInfo Who)
{
  local int i;
  if (WSL!=None){
    WSL.GameEvent(GEvent, Desc, Who);
  }

  for (i = 0; i < ReportPlugins.Length; i++ ){
     ReportPlugins[i].GameEvent(GEvent, Desc, Who);
  }

 	if ( bIgnoreWarmupEvents && !bMatchStarted )
	  return;

  if (GEvent == "NameChange" && Desc ~= Who.OldName)
    return;
  if (IsTeamGame && bReportGameEvents && bUseName){
    if (bDebug) {
      Logf("<!--Game Event-->");
    }
    myHTMLwrapper.WriteSingleReportString(TimeStamp(), myHTMLwrapper.B(CCName(Who))@ClassifyGEvent(GEvent, Desc));
  }
  else if (IsTeamGame && bReportGameEvents && !bUseName){
    if (bDebug) {
      Logf("<!--Game Event-->");
    }
    myHTMLwrapper.WriteSingleReportString(TimeStamp(), ClassifyGEvent(GEvent, Desc));
  }

  bUseName=True;
}

function LogLastSecondSave(int By){
  if (bDebug) {
    Logf("<!-- Last Second Event-->");
  }

	if ( bIgnoreWarmupEvents && !bMatchStarted )
	  return;

  myHTMLwrapper.WriteSingleReportString(TimeStamp(), LastSecondSaveBy@myHTMLwrapper.B(CTName(By)));
}

// ============================================================================
// LogMessage
//
// Logs a said string
// ============================================================================
function LogMessage (PlayerReplicationInfo Who, string Message, optional name Type)
{
  local string Where;
  local string teamstyle;

  if (bClosedReport)
    return;

  Where = GL(Who);

  if (bDebug) {
    Logf("<!-- Say Event-->");
  }

  if (Who.Team.TeamIndex == 0)
  	teamstyle="red";
  else if (Who.Team.TeamIndex == 1)
  	teamstyle="blue";

	if ( bIgnoreWarmupEvents && !bMatchStarted )
	  return;

	if (bColoredMessages){
    If (string(Type) == "TeamSay"){
      myHTMLwrapper.WriteSingleReportString(TimeStamp(), myHTMLwrapper.B(CCName(Who))$"("$Where$")<span class=\"teamsay_"$teamstyle$"\">[teamsay]:"@Message$"</span>");
    }
    else{
      myHTMLwrapper.WriteSingleReportString(TimeStamp(), myHTMLwrapper.B(CCName(Who))$"("$Where$")<span class=\"say\">:"@Message$"</span>");
  	}
  }
  else{
    If (string(Type) == "TeamSay"){
      myHTMLwrapper.WriteSingleReportString(TimeStamp(), myHTMLwrapper.B(CCName(Who))$"("$Where$")[teamsay]:"@Message);
    }
    else{
      myHTMLwrapper.WriteSingleReportString(TimeStamp(), myHTMLwrapper.B(CCName(Who))$"("$Where$"):"@Message);
  	}
  }
}

// ============================================================================
// SpecialEvent
//
// Logs special events like sprees, multikills combo activations, etc.
// ============================================================================

function SpecialEvent(PlayerReplicationInfo Who, string Desc)
{
  local string DecodedEventDesc;
  local int i;

  if (WSL!=None){
    WSL.SpecialEvent(Who, desc);
  }

  for (i = 0; i < ReportPlugins.Length; i++ ){
     ReportPlugins[i].SpecialEvent(who, Desc);
  }

	if ( bIgnoreWarmupEvents && !bMatchStarted )
	  return;

  DecodedEventDesc = DecodeSEDesc(Desc, Who);

  if (bDebug) {
    Logf("<!-- Special Event-->");
  }

  myHTMLwrapper.WriteSingleReportString(TimeStamp(), myHTMLwrapper.B(CCName(Who))@DecodedEventDesc);
}


// ============================================================================
// Logf
//
// Writes a single string into a log
// Each call starts a new line
// ============================================================================

function Logf(string LogString)
{
  if (TempLog!=None && !bTurnOff)
    TempLog.Logf(LogString);
}



// ============================================================================
// RemoveIllegalWindowsCharacters
//
// Takes the given string and removes the character that are not allowed for
// file names in Windows file systems. Called from GetLogFileName
// ============================================================================

function string StripIllegalWindowsCharacters (string result)
{
	ReplaceText(result,".", "-"); 			// Dots
	ReplaceText(result,"*", "-");				// Asterisks
	ReplaceText(result,":", "-");       // Colons
	ReplaceText(result,"|", "-");				// Vertical lines
	ReplaceText(result,"/", "-");				// Slashes
	ReplaceText(result,";", "-");				// Semicolons
	ReplaceText(result,">", "-");				// Less-than signs
	ReplaceText(result,"<", "-");				// Greater-than signs
	ReplaceText(result,"+", "-");				// Pluses
	ReplaceText(result,"?", "-");				// Question Marks
	ReplaceText(result,"\\","-");       // Backslashes (one more is added as the comiler interprest the
																			// only symbols as a mark that the next quotation mark is a part
																			// of the given string, not it's end;
	ReplaceText(result,"\"","_");				// Same for the quotation marks
	return result;
}


// ============================================================================
// GetLogFileName
//
// Takes the log file name mask, replaces vars in it and returns the
// ready-to-use string.
// ============================================================================

function string GetLogFilename()
{
  local string result;

  result = LogFileName;

  // replacing variables in file name mask
  ReplaceText(result, "%G", Level.Game.GameName);
  ReplaceText(result, "%N", Level.Game.GameReplicationInfo.ServerName);
  ReplaceText(result, "%A", Level.Title);
  ReplaceText(result, "%Y", Right("0000"$string(Level.Year), 4));
  ReplaceText(result, "%M", Right("00"$string(Level.Month), 2));
  ReplaceText(result, "%D", Right("00"$string(Level.Day), 2));
  ReplaceText(result, "%H", Right("00"$string(Level.Hour), 2));
  ReplaceText(result, "%I", Right("00"$string(Level.Minute), 2));
  ReplaceText(result, "%W", Right("0"$string(Level.DayOfWeek), 1));
  ReplaceText(result, "%S", Right("00"$string(Level.Second), 2));

  result = StripIllegalWindowsCharacters(result);

  return result;
}


// ============================================================================
// DayOfWeekStr
//
// gets the day name depending on it's number in the week
// ============================================================================

function string DayOfWeekStr(int Num)
{
  if (Num == 1)
    return "Mon";
  if (Num == 2)
    return "Tue";
  if (Num == 3)
    return "Wed";
  if (Num == 4)
    return "Thu";
  if (Num == 5)
    return "Fri";
  if (Num == 6)
    return "Sat";
  if (Num == 7)
    return "Sun";
}


// ============================================================================
// GetMonthStr
//
// gets the month name depending on it's number
// ============================================================================
function string MonthStr(int Num)
{
  if (Num == 1)
    return "Jan";
  if (Num == 2)
    return "Feb";
  if (Num == 3)
    return "Mar";
  if (Num == 4)
    return "Apr";
  if (Num == 5)
    return "May";
  if (Num == 6)
    return "Jun";
  if (Num == 7)
    return "Jul";
  if (Num == 8)
    return "Aug";
  if (Num == 9)
    return "Sep";
  if (Num == 10)
    return "Oct";
  if (Num == 11)
    return "Nov";
  if (Num == 12)
    return "Dec";
}

defaultproperties
{
  myHTMLwrapperClass=class'GameLogger.GameLoggerHTMLwrapper'
  bUseColoredNames=True
  bUseColoredTeamNames=True
  bShowTeamScoreEvents=True
  bShowConnectsDisconnects=True
  bShowSStatus=True
  bDontLogKills=false
  bReportGameEvents=True
  bLogTalks=True
  bDebug=true
  matchplayed="Match played on"
  Player="Player"
  Ping="Ping"
  PL="Packet Loss"
  RulesString="Game Rules"
  GameStartedMsg="Match has started"
  EnteredTheGameMsg="entered the game"
  LeftTheGameMsg="left the game"
  GameEndedMsg="Match has ended"
  ByCompletingObjective="by completing objective"
  ByCapturingFlag="by capturing an enemy flag"
  CapturesTheFlag="by capturing an enemy flag"
  FlagIsCaptured="flag is captured"
  PerformsSuicide="by performing suicide"
  heavyselfdamageM="dealt himself"
  heavyselfdamageF="dealt herself"
  damagestring="% damage"
  bystring="by"
  leavinghim="leaving him"
  leavingher="leaving her"
  damagesstring=" damaged "
  ByHealingPowerNode="by healing Power Node"
  DestroyedEnemyPowerCore="by destroying an enemy Power Core"
  ProtectingTeammate="while protecting a teammate"
  ByKillingAnEnemy="by killing an enemy player"
  ByKillingAnEnemyTeam="for killing a player of an enemy team "
  KillsTeammate="for teamkill/suicide"
  KilledATeammate="by killing a teammate"
  WinsASRound="by winning pair of rounds"
  ScoredGoal="scored a goal"
  ByScoringGoal="by scoring a goal"
  CarriedBall="by successfully carrying ball through the goal"
  ByCarryingBall="by carrying the ball"
  ByAssisting="by assist"
  DOMByScoring="by successfully holding a DOM point"
  NameChange="now known as"
  FlagTaken="takes enemy flag!"
  FlagReturnedTimeout="flag is reset!"
  FlagDropped="drops enemy flag"
  FlagPicked="picks up enemy flag!"
  FlagReturned="returns the flag!"
  FlagCaptured="captures enemy flag!"
  TeamChanged="now plays for"
  CompletedLastObjective=" has completed the final objective!"
  AttackingTeamWon="Attacking team won the round!"
  DefendingTeamWon="Defending team won the round!"
  NewRoundStarted="New assault round has started!"
  NewRoundStarted2="New round has started!"
  CompletedObjective="completes current objective!"
  DestroyedVehicle="destroys enemy vehicle!"
  BallDropped="drops the ball!"
  BallTaken="takes the ball!"
  BallPicked="picks the ball!"
  BallReturnedTimeout="Ball was reset!"
  RedTeam="Red Team"
  Scores="scores"
  Loses="loses"
  BlueTeam="Blue Team"
  LastSecondSaveBy="Last second save by"
  GibbedInTranslocation="dies because of damaged translocator module"
  KilledTypingVictim="kills typing victim"
  DrawsFirstBlood="draws first blood!"
  ArenaName="Arena name:"
  ServerName="Server Name:"
  MutatorsString="Mutators Used:"
  RunsBoosterCombo="activates Booster combo"
  RunsInvisCombo="activates Invis combo"
  RunsBerserkCombo="activates Berserk combo"
  RunsSpeedCombo="activates Speed combo"
  RunsCrateCombo="activates Crate combo"
  RunsMinimeCombo="activates Mini-me combo"
  Admin="Admin"
  Point="point"
  Points="points"
  Points2="points"
  WeaponNames(0)="Shield Gun"
  WeaponNames(1)="Assault Rifle"
  WeaponNames(2)="Biorifle"
  WeaponNames(3)="ASMD Shock Rifle"
  WeaponNames(4)="Link Gun"
  WeaponNames(5)="Minigun"
  WeaponNames(6)="Flak Cannon"
  WeaponNames(7)="Rocket Launcher"
  WeaponNames(8)="Lightning Gun"
  WeaponNames(9)="Sniper Rifle"
  WeaponNames(10)="AVRiL"
  WeaponNames(11)="Grenade Launcher"
  WeaponNames(12)="Mine Layer"
  WeaponNames(13)="Translocator"
  WeaponNames(14)="Redeemer"
  WeaponNames(15)="Ion Painter"
  WeaponNames(16)="Instagib Shock Rifle"
  WeaponNames(17)="unknown weapon."
  PickedWeaponNames(0)="Shield Gun"
  PickedWeaponNames(1)="Assault Rifle"
  PickedWeaponNames(2)="Biorifle"
  PickedWeaponNames(3)="ASMD Shock Rifle"
  PickedWeaponNames(4)="Link Gun"
  PickedWeaponNames(5)="Minigun"
  PickedWeaponNames(6)="Flak Cannon"
  PickedWeaponNames(7)="Rocket Launcher"
  PickedWeaponNames(8)="Lightning Gun"
  PickedWeaponNames(9)="Sniper Rifle"
  PickedWeaponNames(10)="AVRiL"
  PickedWeaponNames(11)="Grenade Launcher"
  PickedWeaponNames(12)="Mine Layer"
  PickedWeaponNames(13)="Translocator"
  PickedWeaponNames(14)="Redeemer"
  PickedWeaponNames(15)="Ion Painter"
  PickedWeaponNames(16)="Instagib Shock Rifle"
  PickedWeaponNames(17)="Target Painter"
  Combo=" succesful shock rifle combo "
  Headshot=" nails "
  headsuffix=" a headshot "
  Burned=" burn "
  Corroded=" corroded "
  Drowned=" drown "
  fell=" fell "
  FellLava=" fell into lava "
  Sburned=" burned "
  Scorroded=" corroded "
  Scrushed=" crushed himself "
  Sdrowned=" drowned "
  Sfell=" left a small crater "
  Sfelllava=" into lava "
  whiletypingM="while he was typing"
  whiletypingF="while she was typing"
  firetype(0)=" primary "
  firetype(1)=" secondary "
  killsstring=" kills "
  suicidestring=" performs suicide "
  With=" with "
  Awith=" with "
  NATprefix=" makes "
  Push=" pushes "
  pushsuffix=" over the edge"
  crush=" crushes "
  ATsuffix="at "
  picks=" picks up "
  lands_on=" made a pancake of "
  runsover=" runs over "
  clips=" clips "
  Gets="makes"
  Keeps="keeps"
  ArmorNames(0)="Shield Pack"
  ArmorNames(1)="Super Shield Pack"
  HealthNames(0)="Health Pack"
  HealthNames(1)="Keg o'Health"
  UDamageName="Double Damage"
  ONSPowerNode_constructed="was constructed by"
  ONSPowerNode_destroyeddueisolation="was destroyed due to isolation"
  ONSPowerNode_destroyed="was destroyed by"
  ONSPowerNode_beingconstructed="is being constructed by"
  ONSPowerNode_disabled="was disabled"
  ONSCoreDestroyed="Power Core was destroyed"
  ONSPowerNodeName="PowerNode"
  RaptorMissile=" Raptor's Air2Air Missile "
  RaptorPlasma=" Raptor's Plasma Guns "
  HBNDSideTurretSec=" Hellbender's side turret secondary "
  HBNDBackTurret=" Hellbender's rear turret "
  MantaPlasma=" Manta's Plasma Guns "
  LinkTurretPlasma=" Link Turret primary "
  LinkTurretBeam=" Link Turret secondary "
  IonTankCannon=" Ion Tank blast "
  IonCannonShot=" Ion Cannon blast "
  LeviBlast=" Leviathan's Ion Blast "
  LeviTurret=" Leviathan's plasma turret "
  MiniTurret=" Minigun Turret "
  VehicleExplosion=" by a nearby vehicle explosion "
  BlownVehicleRK=" with blown vehicle "
  Accident=" had an accident"
  GoliathTurret=" Goliath-mounted chaingun "
  ScorpBlade="'s head off with a Scorpion blade "
  ScorpNet=" Scorpion's Plasma Net "
  HBNDSideTurretCombo=" Hellbender's side turret's shock combo "
  HBNDSideTurretPri=" Hellbender's side turret's primary "
  HFighterLaser=" Human's Fighter's Laser Gun "
  SFighterLaser=" Skaarj's Fighter's Laser Gun "
  HFighterMissile=" Human's Fighter's homing missile "
  SFighterMissile=" Skaarj's Fighter's homing missile "
  GoliathCannon=" Goliath's cannon shot "
  ShockTurret=" Shock Turret's Beam "
  RaptorRK=" in a Raptor "
  MantaHeadClip="'s head off with a Manta"
  IonTankRK=" in an Ion Tank "
  LeviRK=" with an IonTank "
  RaptorPK=" with a Raptor "
  MantaPK=" with a Manta "
  IonTankPK=" with an Ion Tank "
  LeviPK=" with a Leviathan (HOW??!!) "
  HBND_PK=" with a Hellbender "
  ScorpPK=" with a Scorpion "
  GoliathPK=" with a Goliath "
  HBND_RK=" in a Hellbender"
  ScorpRK=" in a Scorpion "
  GoliathRK=" in a Goliath "
  Version="4.0RC3"
  bSortOffOnlySpecs=True
}
