//-----------------------------------------------------------
// A simple CTF-game summary creator, made for my own
// pleasure. Inspired by summaries of ice-hockey games;
//-----------------------------------------------------------
// TODO: Game Winning Poing logging;
// TODO: sofisticated system for writing a short summary
// of the gameflow; (check the templates in TeamScore Event)
//-----------------------------------------------------------
//  by Michael "_Lynx" Sokolkov  2005
//
// if you want to use this code for your own purposes (i.e.
// your own project, I don't mean extensions for GameLogger)
// mail me first: senninha_r@mail.ru
//-----------------------------------------------------------
class GameLoggerCTFSummary extends GameLoggerReportPlugin
  config(GameLogger);

var bool bFirstScore;             // used to find if there were any captures before
var bool bOvertime;               // If the game went in overtime?
var bool bMatchHasStarted;				// Used to track if the match has started, in case if there
																	// a mutator aloowing warm-up and the and it is enabled

// ============================================================================
// team struct
//
// holds information on how well team plays this match. Switches are keeping
// information on score changes and being updated with each capture
// ============================================================================
struct teams{
  var TeamInfo thisTeam;             // Team Info of team to which these switches are related

  var bool bLostlead;                // team had lead, but lost it
  var bool brestoredlead;            // team had the lead, opponent tied, but team went ahead again
  var bool bReturnedlead;            // team was ahead, went behind, then retrieved the lead
  var bool bCameBack;                // team was loosing heavily, but went on a scoring streak and tied at least
  var bool bTookTheLead;             // this team score first and took the lead
  var bool bWasbehind;               // while the previous flag (any) was captured they were behind
  var bool bTiedScore;               // Team tied score on this capture
  var bool bLostleadAndWentBehind;   // Team not only lost lead before this capture, but also went behind

  var int TiedTheScoreTimes;         // How many times this team tied the score
  var int TimesTouchedFlag;          // How many times this team touched the flag. Used for calculating scoring efficiency
};

// ============================================================================
// ScorersAssistants struct
//
// holds information on who captured the flag, who assisted him, what is the
// score after this capture and when it happend
// ============================================================================

struct ScorersAssistants{
  var PlayerReplicationInfo Scorer;          // Player who captures the flag
  var PlayerReplicationInfo FirstAssistant;  // Player who held the flag before the Scorer
  var PlayerReplicationInfo SecondAssistant; // Player who held the flag before the First Assistant

  var string CurrentScore;                   // Score after this capture
  var string TimeStamp;                      // When this capture was made
};

// ============================================================================
// Headings struct:
//
// Simple struct containing all available heading for the summary
// Heading is the simple string with 4 (now) variables you can use:
// %w - winner-team name
// %l - loser-team name
// %c - point name ("capture" actually)
// %p - player who made the last capture of the game
// %o - total score (f.e. if game score was 3:1 thi will be replaced by 4)
// type - type of heading, defines situation when this heading should be used
// The available values for type are:
// 0 - shutout victory (a big score shut-out victory f.e. 7-0 or 5-0)
// 1 - solid (3 point difference - 6-2, 4-1 and so on )
// 2 - hard-fought (1 point difference)
// 3 - willed victory (was losing but retrieved the lead and won) not implemented yet;
// 4 - overtime win
// 5 - willed overtime win (managed to tie the match and win the overtime) not implemented yet;
// 6 - usual win
// 7 - tie
// 8 - 0:0 tie
// 9 - big score match (9:5 and so on, losing team must score more at least 5 goals for this to be used)
//
// Note that the struct and the array, using this struct are configurable, so
// you can add you own headings
// ============================================================================
struct Headings{
  var globalconfig string Heading; // heading string
  var globalconfig int type;       // type of this heading, see explainations above
};

// ============================================================================
// array of Headings struct
// ============================================================================
var globalconfig array<Headings> HeadingsArray;

var globalconfig string SummaryFileName;          // Summary filename mask
var globalconfig string AllowedGameTypeNames[5];  // These are the names of the gametypes that are like CTF
																								  // but difference is not so big, like Instagib CTF or VCTF,
																								  // so you can make this plugin to work with these modes

var localized string SummaryNameString;
var localized string SectionName;
var localized string SummaryNameProperty;

var Teams myTeams[2];                             // TeamInfo
var Controller FirstTouchBlue;                    // temp variable used for ??? deprecated?
var Controller FirstTouchRed;                     // temp variable used for ??? deprecated?

// this is used, as the flag's own Holder and Assists array are emptied, whe the flag returns to it's base. This is done BEFORE the TeamScoreEvent is called
var ScorersAssistants PotentialRedScorers;        // this used to keep current holder of the blue flag and the ones who held it before current holder
var ScorersAssistants PotentialBlueScorers;       // this used to keep current holder of the red flag and the ones who held it before current holder


var PlayerReplicationInfo GameWinningGoalScorer;  // PRI of player who made the capture, after which his team kept the lead till the end of the match
var FileLog TempLog;                              // reference var for created log file
var CTFFlag FlagsInGame[2];                       // reference on both flags, to keep track of their holders and assists arrays

var array<ScorersAssistants> Goals;               // when the capture is made, PotentialRedScorers or PotentialBlueScorers is added to this array

/*
static function FillPlayInfo(PlayInfo PlayInfo)
{
	Super.FillPlayInfo(PlayInfo);

	PlayInfo.AddSetting(default.SectionName,"SummaryFileName",default.SummaryNameString,class'MutGameLogger'.default.Priv,1, "Text", "50");
}
*/
// ============================================================================
// ForThisGametype
//
// Called by the EGameStats when it loads report plugins. If plugin is
// successfully spawned this function is being called in it to check
// if this plugin suits for the current gametype.
//
// Here: if the current game type name is "Capture the Flag" or "Instagib CTF"
// or "Vehicle CTF" or other specified in config, plugin is considered suitable
// for this gametype.
// ============================================================================

function bool ForThisGametype(string GameName){
 if (GameName == "Capture the Flag" ||
 	 GameName == "Instagib CTF" ||
	 GameName == "Vehicle CTF" ||
	 (AllowedGameTypeNames[0] != "" && GameName == AllowedGameTypeNames[0]) ||
	 (AllowedGameTypeNames[1] != "" && GameName == AllowedGameTypeNames[1]) ||
	 (AllowedGameTypeNames[2] != "" && GameName == AllowedGameTypeNames[2]) ||
	 (AllowedGameTypeNames[3] != "" && GameName == AllowedGameTypeNames[3]) ||
	 (AllowedGameTypeNames[4] != "" && GameName == AllowedGameTypeNames[4])
	 )
  return true;
 else
  return false;
}


// ============================================================================
// InitializePlugin
//
// Sets initial values of some variables and creates the log. All that stuff
// was initially performed in PostBeginPlay, but I needed it to performed not ]
// when the class is created, but only when I needed it.
// ============================================================================

function InitializePlugin(){

  bFirstScore = true;

  // Starting the log that will be the summary
  TempLog = spawn(class 'FileLog');
  if (TempLog != None){
    TempLog.OpenLog(GetLogFilename(), "html");
  }
  else {
    Warn("Could not create output file");
  }
}


// ============================================================================
// StartGame
//
// This one simply enables the logging of all events, so all that happened
// before the game started won't get into the summary (i.e. in TTM2004 or
// UTComp warm up)
// ============================================================================
function StartGame(){
  bMatchHasStarted = True;
	myTeams[0].TimesTouchedFlag = 0;
	myTeams[1].TimesTouchedFlag = 0;
}


// ============================================================================
// Timer
//
// Tracks if the flag is at his placeholder and if not keeps trackinh who holds
// it and if there any assistants for this flag. This is done so because the
// flag simply resets this information JUST before reporting a capture, so we
// can't check them in TeamScoreEvent. So we recreate this vars here and if the
// flag is not on the base we update them with approprite vars form it, and
// when someone scores just write down appropriate values from local copies
// ============================================================================

function Timer(){
  local int i;
  local GameReplicationInfo GRI;

	if (!bMatchHasStarted)// Don't do anything before the match starts
	  return;

  GRI = Level.Game.GameReplicationInfo;

  if (FlagsInGame[1].Location != FlagsInGame[1].HomeBase.Location){
  // do the below stuff ONLY when flag is not in his placeholder

    // getting the current flag holder
    for (i=0; i<GRI.PRIArray.Length; i++){
      if ( GRI.PRIArray[i].HasFlag == FlagsInGame[1]){
        PotentialBlueScorers.Scorer = GRI.PRIArray[i];
      }
    }

    // getting assists
    // note: the last assist == scorer if scorer != firstTouch
    if (FlagsInGame[1].Assists.length >= 3){ // there are at least two assistants
      PotentialBlueScorers.FirstAssistant  = FlagsInGame[1].Assists[FlagsInGame[1].Assists.length-2].PlayerReplicationInfo;
      PotentialBlueScorers.SecondAssistant = FlagsInGame[1].Assists[FlagsInGame[1].Assists.length-3].PlayerReplicationInfo;
    }

    else if (FlagsInGame[1].Assists.length == 2){
      PotentialBlueScorers.FirstAssistant = FlagsInGame[1].Assists[0].PlayerReplicationInfo;
      if (FlagsInGame[1].FirstTouch.PlayerReplicationInfo != PotentialBlueScorers.Scorer)
        PotentialBlueScorers.SecondAssistant = FlagsInGame[1].FirstTouch.PlayerReplicationInfo;
    }

    else if (FlagsInGame[1].Assists.length == 1){
      if (FlagsInGame[1].FirstTouch.PlayerReplicationInfo != PotentialBlueScorers.Scorer)
        PotentialBlueScorers.FirstAssistant = FlagsInGame[1].FirstTouch.PlayerReplicationInfo;
    }
    // just a check, so one player won't be mentioned in assistants twice
    if (PotentialBlueScorers.FirstAssistant == PotentialBlueScorers.SecondAssistant)
      PotentialBlueScorers.SecondAssistant = None;
  }

  // the same as the above, but for another flag
  if (FlagsInGame[0].Location != FlagsInGame[0].HomeBase.Location){

    for (i=0; i<GRI.PRIArray.Length; i++){
      if ( GRI.PRIArray[i].HasFlag == FlagsInGame[0]){
        PotentialRedScorers.Scorer = GRI.PRIArray[i];
      }
    }

    // note: the last assist == scorer if scorer != firstTouch
    if (FlagsInGame[0].Assists.length >= 3){
      PotentialRedScorers.FirstAssistant  = FlagsInGame[0].Assists[FlagsInGame[0].Assists.length-2].PlayerReplicationInfo;
      PotentialRedScorers.SecondAssistant = FlagsInGame[0].Assists[FlagsInGame[0].Assists.length-3].PlayerReplicationInfo;
    }

    else if (FlagsInGame[0].Assists.length == 2){
      PotentialRedScorers.FirstAssistant = FlagsInGame[0].Assists[0].PlayerReplicationInfo;
      if (FlagsInGame[0].FirstTouch.PlayerReplicationInfo != PotentialRedScorers.Scorer)
        PotentialRedScorers.SecondAssistant = FlagsInGame[0].FirstTouch.PlayerReplicationInfo;
    }

    else if (FlagsInGame[0].Assists.length == 1){
      if (FlagsInGame[0].FirstTouch.PlayerReplicationInfo != PotentialRedScorers.Scorer)
        PotentialRedScorers.FirstAssistant = FlagsInGame[0].FirstTouch.PlayerReplicationInfo;
    }

    if (PotentialRedScorers.FirstAssistant == PotentialRedScorers.SecondAssistant)
      PotentialRedScorers.SecondAssistant = None;
  }

  // overtime check
  if (Level.Game.bOverTime)
   bOvertime = true;
}

// ============================================================================
// GameEvent
//
// Being called from the GameLogger.EgameStats GameEvent function
//
// If the event is "flag_taken" (flag was taken from it's base if from
// the ground is "flag_pickup"), we're increasing the TimesTouchedFlag for the
// appropriate team, so we can get total number of attempts to steal the flag
// by that team.
//
// Note: for the "flag_taken" event as a desc, CTFFlag, which sends this GEvent
// to GameStats, passes the index of team who owns that taken flag, not the one
// who took it.
// ============================================================================
function GameEvent(string GEvent, string Desc, PlayerReplicationInfo Who){
  if (GEvent == "flag_taken" && bMatchHasStarted){
    if (Desc == "1"){ // see note above
      myTeams[0].TimesTouchedFlag++;
    }
    else {
      myTeams[1].TimesTouchedFlag++;
    }
  }
}

// ============================================================================
// TeamScoreEvent
//
// Being called from the GameLogger.EgameStats TeamScoreEvent function
//
// Does all the scoring related stuff - updates the Goals array, myTeams array
// switches That's a "heart" of theis class.
//
// Under Construction...
// ============================================================================
function TeamScoreEvent(int team, float points, string desc){
  local int opponent, l;

	if (!bMatchHasStarted)// Don't log captures before the match starts
	  return;

  // getting opponent index
  if (team == 0){
    opponent = 1;
  }
  else{
    opponent = 0;
  }

  // that's the first capture of the game; setting initial values for both teams
  if (bFirstScore){
    myTeams[team].bTookTheLead   = True;
    myTeams[opponent].bWasbehind = true;

    bFirstScore = false;

    // someone scored first goal of the game. if his team will keep the lead,
    // he'll be the GWG scorer;
    if (team == 1){
      GameWinningGoalScorer = PotentialRedScorers.Scorer;
    }
    else{
      GameWinningGoalScorer = PotentialBlueScorers.Scorer;
    }
  }
// ===========================================================================
// This section of function is under construction;
// Actually only 12 or so lines relating to determining GWG scorer are
// working here now
// ===========================================================================
  // score just tied
  if (myTeams[team].thisTeam.Score == myTeams[opponent].thisTeam.Score){
    GameWinningGoalScorer = None;// score is tied, GWG race starts again;

    if (myTeams[team].bWasBehind){
      myTeams[opponent].bLostlead     = True;
      myTeams[opponent].brestoredlead = false;
      myTeams[opponent].bReturnedlead = false;
      myTeams[team].bTiedScore=True;
      myTeams[team].TiedTheScoreTimes += 1;
    }
    else{
      myTeams[team].bLostlead=True;
      myTeams[team].brestoredlead=false;
      myTeams[opponent].bTiedScore=True;
      myTeams[opponent].TiedTheScoreTimes += 1;
    }
  }

  // WORKING GUIDE. To save time on scrolling to the heading and back
  //  var bool bLostlead;
  //  var bool brestoredlead;
  //  var bool bReturnedlead;
  //  var bool bCameBack;
  //  var bool bTookTheLead;
  //  var bool bWasbehind;
  //  var bool bTiedScore;
  //  var bool bLostleadAndWentBehind;
  //  var int TiedTheScoreTimes;

  if (myTeams[team].thisTeam.Score > myTeams[opponent].thisTeam.Score){
    if (myTeams[opponent].bTiedScore){
      myTeams[team].brestoredlead  = true;
      myTeams[opponent].bTiedScore = false;

      // so, if this opposite team tied the score but scoring team went ahead again,
      // the one who scored is most likely to became GWG scorer;
      if (team == 1){
        GameWinningGoalScorer = PotentialRedScorers.Scorer;
      }
      else{
        GameWinningGoalScorer = PotentialBlueScorers.Scorer;
      }
    }
    else if (myTeams[team].bTiedScore){
      myTeams[opponent].bLostleadAndWentBehind = True;
      myTeams[opponent].brestoredlead          = false;
      myTeams[opponent].bReturnedlead          = false;

      myTeams[team].bTiedScore = false;
      myTeams[team].bWasBehind = false;
      // if this team tied the score but again went ahead, the one who scored is
      // most likely to became GWG scorer;
      if (team == 1){
        GameWinningGoalScorer = PotentialRedScorers.Scorer;
      }
      else{
        GameWinningGoalScorer = PotentialBlueScorers.Scorer;
      }
    }
    else if (myTeams[team].bLostlead){
      myTeams[team].bLostlead      = false;
      myTeams[team].brestoredlead  = true;
      myTeams[opponent].bTiedScore = false;
    }
     else if (myTeams[team].bLostleadAndWentBehind){
      myTeams[team].bLostleadAndWentBehind = false;
      myTeams[team].bReturnedlead          = true;
      myTeams[opponent].bLostleadAndWentBehind = true;
      myTeams[opponent].bLostLead              = True;
    }
  }
// ===========================================================================
// End of under construction section
//
// ===========================================================================

   // Filling the goals array with the current appropriate PotentialScorers array
    if (CTFGame(Level.Game) != None){
      if(team == 1){
        l = Goals.Length;

        Goals.Length          = l + 1;
        Goals[l]              = PotentialRedScorers;
        Goals[l].TimeStamp    = baseLogger.TimeStamp(); //
        Goals[l].CurrentScore = int(myTeams[0].thisTeam.Score)$":"$int(myTeams[1].thisTeam.Score);

        PotentialRedScorers.Scorer          = None; // so we've set the Goals to PotentialScorers
        PotentialRedScorers.FirstAssistant  = None; // now we can empty all values
        PotentialRedScorers.SecondAssistant = None; // just to avoid any problems
      }
      else{
        l = Goals.Length;
        Goals.Length          = l + 1;
        Goals[l]              = PotentialBlueScorers;
        Goals[l].TimeStamp    = baseLogger.TimeStamp();
        Goals[l].CurrentScore = int(myTeams[0].thisTeam.Score)$":"$int(myTeams[1].thisTeam.Score);

        PotentialBlueScorers.Scorer          = None;
        PotentialBlueScorers.FirstAssistant  = None;
        PotentialBlueScorers.SecondAssistant = None;
    }
  }


}


// ============================================================================
// NewGame
//
// Sets some variables and writes the summary title, style, finds flags to
// track and enables Timer event
// ============================================================================

function NewGame(string ngTitle, string ngAuthor, string ngGameGameName, string siServerName, string siAdminName, string siAdminEmail, string siMOTD){
  local string blueTeamName;
  local string redTeamName;

  local int FragLimit;
  local int TimeLimit;
  local int MaxSpectators;
  local int NumBots;
  local int MaxPlayers;

  local bool bAllowBehindView;
  local bool bAllowWeaponThrowing;
  local bool bWeaponStay;
  local bool bWeaponShouldViewShake;

  local float GameSpeed;

  local CTFFlag testFlag;

  myTeams[0].thisTeam = Level.Game.GameReplicationInfo.Teams[0];
  myTeams[1].thisTeam = Level.Game.GameReplicationInfo.Teams[1];

  myTeams[0].TiedTheScoreTimes = 0;
  myTeams[0].TimesTouchedFlag  = 0;
  myTeams[1].TiedTheScoreTimes = 0;
  myTeams[1].TimesTouchedFlag  = 0;

  FragLimit              = Level.Game.GoalScore;
  TimeLimit              = Level.Game.TimeLimit;
  GameSpeed              = Level.Game.GameSpeed;
  MaxSpectators          = Level.Game.MaxSpectators;
  NumBots                = Level.Game.NumBots;
  MaxPlayers             = Level.Game.MaxPlayers;
  bAllowBehindView       = Level.Game.bAllowBehindView;
  bAllowWeaponThrowing   = Level.Game.bAllowWeaponThrowing;
  bWeaponShouldViewShake = Level.Game.bWeaponShouldViewShake;
  bWeaponStay            = Level.Game.bWeaponStay;

  blueTeamName = class'MutGameLogger'.default.MyBlueTeamName;
  redTeamname  = class'MutGameLogger'.default.MyRedTeamName;

  Logf("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">");
  Logf("<HTML><HEAD>");
  Logf("<TITLE>Summary of CTF game played on"@BaseLogger.FullTimeDate()@":"@ngGameGameName@" at "@ngTitle@"</title>");
  Logf("<meta name=\"Generator\" content=\"UnrealEngine2 build "$Level.EngineVersion$" - exporter: "$string(self.Class)$", a Summary creator v0.9 for "$string(baseLogger.Class)$" v."$BaseLogger.version@"by _Lynx\">");

  Logf("<!--");
  Logf("// Generated by CTF Summary Plugin v0.9b");
  Logf("// by Michael \"_Lynx\" Sokolkov  2005");
  Logf("// senninha_r@mail.ru");
  Logf("-->");

  // writing down styles
  Logf("<style type=\"text/css\">");
  // usual text
  Logf("   .text{");
  Logf(" font-family : Verdana;");
  Logf(" font-size : 11px;");
  Logf(" }");
  // smaller text
  Logf(" .textX{");
  Logf(" font-family : Verdana;");
  Logf(" font-size : 10px;");
  Logf(" }");
  // heading
  Logf(" .heading{");
  Logf(" font-family : Verdana;");
  Logf(" font-size : 14px;");
  Logf(" }");
  // links - usual
  Logf(" A:link.score {");
  Logf(" COLOR: black;");
  Logf(" font-weight: normal;");
  Logf(" font-style: normal;");
  Logf(" font-variant: normal;");
  Logf(" text-decoration: none;");
  Logf(" }");
  // links - visited link
  Logf(" A:visited.score {");
  Logf(" color: black;");
  Logf(" font-weight: normal;");
  Logf(" font-style: normal;");
  Logf(" font-variant: normal;");
  Logf(" text-decoration: none;");
  Logf(" }");
  // links - links when cursor is above it
  Logf(" A:hover.score {");
  Logf(" COLOR: navy;");
  Logf(" font-weight: bold;");
  Logf(" font-style: normal;");
  Logf(" font-variant: normal;");
  Logf(" text-decoration: underline;");
  Logf(" }");
  // a hidden table for scores;
  Logf(" TABLE.common{");
  Logf(" background-color : White;");
  Logf(" border-bottom : 0px solid Black;");
  Logf(" border-left : 0px solid Black;");
  Logf(" border-right : 0px solid Black;");
  Logf(" border-top : 0px solid Black;");
  Logf(" font-family : Verdana;");
  Logf(" font-size : 10px;");
  Logf(" }");
  Logf("</style>");
  // end of embedded CSS

  Logf("</head>");
  Logf("<body bgcolor=white rightmargin=\"10\" leftmargin=\"10\" topmargin=\"10\" text=\"#000000\" class=textx>");

  // looking for flags and assigning them to the reference vars. This is done
  // here, since if done from PostBeginPlay, no flags are found
  ForEach AllActors(Class'CTFFlag', testFlag){
    log("found CTFFlag:"@testFlag);

     if (TestFlag.TeamNum == 0){
       FlagsInGame[0] = TestFlag;
     }
     else if (TestFlag.TeamNum == 1){
       FlagsInGame[1] = TestFlag;
     }
  }

  // Flags are assigned, now starting the timer, so, we can track their state
  SetTimer(0.5, True);
}


// ============================================================================
// CreateHeading
//
// selects the heading from the headings array depending on the final score,
// replaces variables in it and returns the ready-to-use string
// ============================================================================
function string CreateHeading(GamereplicationInfo GRI, String LastScorer){

  local teamInfo winners;
  local teamInfo losers;

  local bool bShutOut;       // Only one team scored in this game - shut-out
  local bool bSolid;         // A solid victory with 3 points gap
  local bool bhardfought;    // A tough game when only one point difference in the end
  local bool bDraw;          // resultive draw (1:1. 4:4, etc)
  local bool bZeroDraw;      // 0:0 draw
//  local bool bWilled;        // match won be a team that was losing in the beginning
  local bool bAllOutAttack;  // both teams played with little or no defence - huge score like 7:8

  local string result;       // the created heading will be returned via this var
  local string point;        // point for this gametype. to replace %p in headig
  local string TotalScore;   // this string will replace the %o in headings

  local int i;
  local int l;
  local int headnum;

  local array<string> SuitableHeadings; // The array with headings of the type that suits for this game score

  point = "capture";

  // defining who's who
  if (GRI.Teams[0].Score > GRI.Teams[1].Score){
    winners = GRI.Teams[0];
    losers = GRI.Teams[1];
  }
  else{
    winners = GRI.Teams[1];
    losers = GRI.Teams[0];
  }

  // losers scored 0, winners 5 and more (f.e. 0:5, 0:7, 0:8)
  if (losers.Score == 0 && winners.Score > 4)
   bShutOut = true;

  // losers scored at least 5. and winners - 6+ since it's a win (f.e. 5:6, 7:11, 5:14)
  if (( int(winners.Score - losers.Score) <= 2 || losers.Score >= 5) && int(winners.Score + losers.Score) >= 10)
   bAllOutAttack = true;

  // losers score any less than 5, winners - 3+ more than losers (f.e. 1:4, 4:7, 2:8)
  if ((int(winners.Score - losers.Score) >= 3) && !bAllOutAttack)
   bSolid = true;

  // if the score is not huge and the difference is only 1 point (f.e. 0:1, 3:4, 4:5)
  if (int(winners.Score - losers.Score) == 1 && !bAllOutAttack)
   bhardfought=True;

  // resultive draw (f.e. 1:1, 4:4, 5:5). for a forced end of game
  if (GRI.Teams[0].Score == GRI.Teams[1].Score)
   bDraw = True;

  // 0:0 draw (if someone forcibly stopped before someone scored)
  if ( (GRI.Teams[0].Score == GRI.Teams[1].Score) && int(GRI.Teams[0].Score) == 0)
   bZeroDraw = True;

  // Looking through available headings and picking the ones of the matching type;
  if (bShutOut){
    for (i = 0; i < HeadingsArray.Length; i++){
      if (HeadingsArray[i].type == 0){
        l = SuitableHeadings.Length;
        SuitableHeadings.Length = l + 1;
        SuitableHeadings[l] = HeadingsArray[i].Heading;
      }
    }
  }

  if (!bShutOut && bSolid){
    for (i = 0; i < HeadingsArray.Length; i++){
      if (HeadingsArray[i].type == 1){
        l = SuitableHeadings.Length;
        SuitableHeadings.Length = l + 1;
        SuitableHeadings[l] = HeadingsArray[i].Heading;
      }
    }
  }

  if (bhardfought && !bOvertime){
    for (i = 0; i < HeadingsArray.Length; i++){
      if (HeadingsArray[i].type == 2){
        l = SuitableHeadings.Length;
        SuitableHeadings.Length = l + 1;
        SuitableHeadings[l] = HeadingsArray[i].Heading;
      }
    }
  }

  if (bOverTime){
    for (i = 0; i < HeadingsArray.Length; i++){
      if (HeadingsArray[i].type == 4){
        l = SuitableHeadings.Length;
        SuitableHeadings.Length = l + 1;
        SuitableHeadings[l] = HeadingsArray[i].Heading;
      }
    }
  }

  if (bDraw && !bZeroDraw){
    for (i = 0; i < HeadingsArray.Length; i++){
      if (HeadingsArray[i].type == 7){
        l = SuitableHeadings.Length;
        SuitableHeadings.Length = l + 1;
        SuitableHeadings[l] = HeadingsArray[i].Heading;
      }
    }
  }

  if (bZeroDraw){
    for (i = 0; i < HeadingsArray.Length; i++){
      if (HeadingsArray[i].type == 8){
        l = SuitableHeadings.Length;
        SuitableHeadings.Length = l + 1;
        SuitableHeadings[l] = HeadingsArray[i].Heading;
      }
    }
  }

  if (bAllOutAttack){
    for (i = 0; i < HeadingsArray.Length; i++){
      if (HeadingsArray[i].type == 9){
        l = SuitableHeadings.Length;
        SuitableHeadings.Length = l + 1;
        SuitableHeadings[l] = HeadingsArray[i].Heading;
      }
    }
  }

  // a usual win like 2:0
  if (!bDraw && !bZeroDraw && !Level.Game.bOverTime && !bhardfought && !bShutOut && !bSolid){
    for (i = 0; i < HeadingsArray.Length; i++){
      if (HeadingsArray[i].type == 6){
        l = SuitableHeadings.Length;
        SuitableHeadings.Length = l + 1;
        SuitableHeadings[l] = HeadingsArray[i].Heading;
      }
    }
  }

  headnum    = Rand(SuitableHeadings.Length-1);       // randomly selectinga a header amnong picked ones
  result     = SuitableHeadings[headnum];             // setting result to that heading
  TotalScore = string(winners.Score + losers.Score);  // setting the TotalScore

  // replacing variables in the heading
  ReplaceText(result, "%w", winners.TeamName);
  ReplaceText(result, "%l", losers.TeamName);
  ReplaceText(result, "%p", point);
  ReplaceText(result, "%c", LastScorer);
  ReplaceText(result, "%o", TotalScore);

  // returning the finished heading
  return result;
}

// ============================================================================
// AfterReportEnd
//
// Called from EGameStats when the game ended
// Writes down the whole summary
// ============================================================================

function AfterReportEnd(array<PlayerReplicationInfo> PRIs, GameReplicationInfo GRI, string reason){
  local int i;

  local string RedLineup;
  local string BlueLineUp;
  local string Points;
  local string LastScorer;

  // stopping timer to avoid accessed nones, which happen if flag of one of the
  // teams is being carried when the match ends and carrier exits
  SetTimer(0.1, false);

  LastScorer = Goals[Goals.Length-1].Scorer.PlayerName;

  points = "Capture";// "s" will be $-ed in the string

  Logf("<div align=left class=heading>");
  Logf(CreateHeading(GRI, LastScorer));
  Logf("</div>");
  Logf("<BR>");
  Logf(TodayHourServerMap());
  Logf("<BR>");
  Logf("<BR>");
  Logf("<div align=left class=text><strong>"$myTeams[0].thisTeam.TeamName@"-"@myTeams[1].thisTeam.TeamName@"-"@int(myTeams[0].thisTeam.Score)$":"$int(myTeams[1].thisTeam.Score));

  // If the game ended in overtime, add OT after the score
  if (bOvertime){
    Logf(" OT");
  }

  Logf("</strong></div>");
  Logf("<BR>");
  Logf("<strong>"$Points$"s:</strong>");
  Logf("<Table class=common>");
  for (i=0; i < Goals.Length; i++){
    Logf("<TR>");

    if (Goals[i].FirstAssistant == Goals[i].Scorer)
      Goals[i].FirstAssistant = None;

    if (Goals[i].SecondAssistant == Goals[i].Scorer)
      Goals[i].SecondAssistant = None;

    if (Goals[i].FirstAssistant != None && Goals[i].SecondAssistant != None){
      Logf("<TD nowrap>"$Goals[i].Timestamp$"</TD><TD><strong>"$Goals[i].CurrentScore$"</strong>"@Goals[i].Scorer.PlayerName$" ("$Goals[i].FirstAssistant.PlayerName$", "$Goals[i].SecondAssistant.PlayerName$")</TD>");
    }
    else if (Goals[i].FirstAssistant != None){
      Logf("<TD nowrap>"$Goals[i].Timestamp$"</TD><TD><strong>"$Goals[i].CurrentScore$"</strong>"@Goals[i].Scorer.PlayerName$" ("$Goals[i].FirstAssistant.PlayerName$")</TD>");
    }
    else if (Goals[i].SecondAssistant != None){
      Logf("<TD nowrap>"$Goals[i].Timestamp$"</TD><TD><strong>"$Goals[i].CurrentScore$"</strong>"@Goals[i].Scorer.PlayerName$" ("$Goals[i].SecondAssistant.PlayerName$")</TD>");
    }
    else{
      Logf("<TD nowrap>"$Goals[i].Timestamp$"</TD><TD><strong>"$Goals[i].CurrentScore$"</strong>"@Goals[i].Scorer.PlayerName$" (unassisted)</TD>");
    }
    Logf("</TR>");
  }
  Logf("</TABLE");
  Logf("<BR>");

  Logf("<hr align=\"left\" color=\"#000000\" noshade size=\"1\" width=\"500\">");

  // ==========================================================================
  // Stats section
  // ==========================================================================
  Logf("Stats:<BR>");

  Logf("Scoring attempts:");
  Logf(int(myTeams[0].thisTeam.Score)$"/"$myTeams[0].TimesTouchedFlag$"("$GetPercentage(myTeams[0].thisTeam.Score, float(myTeams[0].TimesTouchedFlag))$") - "$int(myTeams[1].thisTeam.Score)$"/"$myTeams[1].TimesTouchedFlag$"("$GetPercentage(myTeams[1].thisTeam.Score, float(myTeams[1].TimesTouchedFlag))$")<BR>");

  if (GameWinningGoalScorer != None){
    Logf("Game Winning "$Points$" made by: ");
    Logf(GameWinningGoalScorer.PlayerName);
    Logf("<BR>");
  }

  Logf("Best Frag of the match:"@GetBestFrag(GRI));
  Logf("<BR>");

//  Logf("Darwin award:"@GetBestFrag(GRI));
//  Logf("<BR>");

  Logf("Players' scores:"@GetPlayersScores(GRI));
  Logf("<BR>");

  Logf("Top scorer:"@GetHighestScore(GRI));
  Logf("<BR>");

  Logf("Most Flags returned:"@GetMostFlagReturns(GRI));
  Logf("<BR>");

  // ==========================================================================
  // Lineups and 3 star selection
  // ==========================================================================
  Logf("<hr align=\"left\" color=\"#000000\" noshade size=\"1\" width=\"500\">");

  Logf("Lineups:");
  Logf("<BR>");
  Logf("<strong>"$myTeams[0].thisTeam.TeamName$":</strong>  ");

  for (i = 0; i < PRIs.Length; i++){
     if (PRIs[i].Team != None && PRIs[i].Team.teamIndex == 0){
       if (!PRIs[i].bIsSpectator && !PRIs[i].bOnlySpectator)
       redLineUp = redLineUp@PRIs[i].PlayerName$",";
     }
  }

  Logf(redLineUp);

  Logf("<BR>");
  Logf("<BR>");
  Logf("<strong>"$myTeams[1].thisTeam.TeamName$":</strong>  ");

  for (i = 0; i < PRIs.Length; i++){
     if (PRIs[i].Team != None && PRIs[i].Team.teamIndex == 1){
       if (!PRIs[i].bIsSpectator && !PRIs[i].bOnlySpectator)
       BlueLineUp = BlueLineUp@PRIs[i].PlayerName$",";
     }
  }

  Logf(blueLineUp);

  Logf("<BR>");
  Logf("<BR>");
  Logf("Three stars:"@GetThreeStars(GRI));

  Logf("<BR>");
  Logf("<hr align=\"left\" color=\"#000000\" noshade size=\"1\" width=\"500\">");

  Logf("Summary by CTF-game summary plugin for <a href=\"http://unreal-pz.narod.ru/en/matchlogger.html\" class=score>GameLogger</a>.<br>");
  Logf("By Michael \"_Lynx\" Sokolkov  2005");

  Logf("</BODY>");
  Logf("</HTML>");

  // All is done, closing the log
  Shutdown();
}

// ============================================================================
// GetPercentage
//
//  A simple function to calculate how much percent of 'whole' 'part' is
// ============================================================================
function string GetPercentage(float part, float whole){
  local string result;
  local float hundreth;

  hundreth = whole/100;
  result = string(int(int(part)/hundreth));
  return result$"%";
}

// ============================================================================
//  Get Best Frag
//
// No, it doesn't describe the best and most beautiful frag of the match. It
// just returns the name of the player who _was_ best frag, in other words, who
// died more than others, being an excellent frag. Sometimes, they even score
// more than others =). Suicides are not counted. They're substracted.
// ============================================================================
function string GetBestFrag(GameReplicationInfo GRI){
  local int i, j;
  local array<PlayerReplicationInfo> PRIs;
  local PlayerReplicationInfo PRI,t;

  for (i=0;i<GRI.PRIArray.Length;i++)
  {
    PRI = GRI.PRIArray[i];

    PRIs.Length = PRIs.Length+1;
    for (j=0;j<Pris.Length-1;j++)
    {
      if ((PRIs[j].Deaths - TeamPLayerReplicationInfo(PRIs[j]).Suicides) < (PRI.Deaths - TeamPLayerReplicationInfo(PRI).Suicides) ||
           ((PRIs[j].Deaths - TeamPLayerReplicationInfo(PRIs[j]).Suicides) == (PRI.Deaths - TeamPLayerReplicationInfo(PRI).Suicides) && PRIs[j].Score > PRI.Score)
      )
      {
        t = PRIs[j];
        PRIs[j] = PRI;
        PRI = t;
      }
    }
    PRIs[j] = PRI;
  }

  for (i = 0; i < PRIs.Length; i++){
    if ( Controller(PRIs[i].Owner).IsA('MessagingSpectator') || Controller(PRIs[i].Owner).IsA('DemoRecSpectator') || Controller(PRIs[i].Owner).IsInState('Spectating')){
      if (bDebug){
        Log("Found a"@PRIs[i].PlayerName@"thinking he's a BestFrag. Removing that f*cker!"); //sorry for the language, but I got tired of seing WebAdmin as the First star
      }
      PRIs.Remove(i, 1);
      i = i - 1; // so the loop won't skip the next entry.
      }
  }

  // who the hell made deaths a float variable?
  return PRIs[0].PlayerName@"("$PRIs[0].Team.TeamName$") - was fragged"@int(PRIs[0].Deaths)@"times.";
}


// ============================================================================
//  GetHighestScore
//
// Finds the player with highest score
// ============================================================================

function string GetHighestScore(GameReplicationInfo GRI){
  local int i, j;
  local array<PlayerReplicationInfo> PRIs;
  local array<PlayerReplicationInfo> localPRIarray;
  local PlayerReplicationInfo PRI,t;

  localPRIarray = GRI.PRIArray;

  for (i = 0; i < localPRIarray.Length; i++){
    if ( Controller(localPRIarray[i].Owner).IsA('MessagingSpectator') || Controller(localPRIarray[i].Owner).IsA('DemoRecSpectator') || Controller(localPRIarray[i].Owner).IsInState('Spectating')){
      if (bDebug){
        Log("Found a"@localPRIarray[i].PlayerName@"thinking he scored somewhere. Removing that f*cker!"); //sorry for the language, but I got tired of seing WebAdmin as the First star
      }
      localPRIarray.Remove(i, 1);
      i = i - 1; // so the loop won't skip the next entry
      }
  }

 for (i=0;i<localPRIarray.Length;i++)
  {
    PRI = localPRIarray[i];

    PRIs.Length = PRIs.Length+1;
    for (j=0;j<Pris.Length-1;j++)
    {
      if (PRIs[j].Score < PRI.Score ||
           (PRIs[j].Score == PRI.Score && PRIs[j].Deaths > PRI.Deaths)
      )
      {
        t = PRIs[j];
        PRIs[j] = PRI;
        PRI = t;
      }
    }
    PRIs[j] = PRI;
  }

  return PRIs[0].PlayerName@"("$PRIs[0].Team.TeamName$") - with score of"@int(PRIs[0].Score)$".";
}


// ============================================================================
//  GetPlayersScores
//
// Calculates how big is the sum of players'score for each team and returns
// this value as a string;
// ============================================================================

function string GetPlayersScores(GameReplicationInfo GRI){
  local int i;
  local array<PlayerReplicationInfo> localPRIarray;
  local float BluesScore;
  local float RedsScore;

  localPRIarray = GRI.PRIArray;

  for (i = 0; i < localPRIarray.Length; i++){
    if ( Controller(localPRIarray[i].Owner).IsA('MessagingSpectator') || Controller(localPRIarray[i].Owner).IsA('DemoRecSpectator') || Controller(localPRIarray[i].Owner).IsInState('Spectating')){
      if (bDebug){
        Log("Found that"@localPRIarray[i].PlayerName@" again. This time in teams' scores. Is there any way to ban that f*cker?!"); //sorry for the language, but I got tired of seing WebAdmin as the First star
      }
      localPRIarray.Remove(i, 1);
      i = i - 1; // so the loop won't skip the next entry
      }
  }

  for (i=0;i<localPRIarray.Length;i++)
  {
    if (localPRIarray[i].Team.TeamIndex == 0)
      RedsScore = RedsScore + localPRIarray[i].Score;
    else if (localPRIarray[i].Team.TeamIndex == 1){
      BluesScore = BluesScore + localPRIarray[i].Score;
    }
  }

  return int(redsscore)@"-"@int(bluesscore);
}


// ============================================================================
//  Get Best Frag
//
// No it doesn't describe the best and most beautidul frag of the match. It
// just returns the name of the player who was best frag, in other words, who
// died more than others
// ============================================================================

function string GetMostFlagReturns(GameReplicationInfo GRI){
  local int i, j;
  local array<PlayerReplicationInfo> PRIs;
  local PlayerReplicationInfo PRI,t;

  for (i=0;i<GRI.PRIArray.Length;i++)
  {
    PRI = GRI.PRIArray[i];

    PRIs.Length = PRIs.Length+1;
    for (j=0;j<Pris.Length-1;j++)
    {
      if (TeamPlayerReplicationInfo(PRIs[j]).FlagReturns < TeamPlayerReplicationInfo(PRI).FlagReturns ||
           (TeamPlayerReplicationInfo(PRIs[j]).FlagReturns == TeamPlayerReplicationInfo(PRI).FlagReturns && PRIs[j].Score < PRI.Score)
      )
      {
        t = PRIs[j];
        PRIs[j] = PRI;
        PRI = t;
      }
    }
    PRIs[j] = PRI;
  }

  for (i = 0; i < PRIs.Length; i++){
    if ( Controller(PRIs[i].Owner).IsA('MessagingSpectator') || Controller(PRIs[i].Owner).IsA('DemoRecSpectator') || Controller(PRIs[i].Owner).IsInState('Spectating')){
      if (bDebug){
        Log("Found a"@PRIs[i].PlayerName$" thinking he returned most flags. Removing that f*cker!"); //sorry for the language, but I got tired of seing WebAdmin as the First star
      }
      PRIs.Remove(i, 1);
      i = i - 1; // so the loop won't skip the next entry.
      }
  }

  return PRIs[0].PlayerName@"("$PRIs[0].Team.TeamName$") - returned"@TeamPlayerReplicationInfo(PRIs[0]).FlagReturns@"flags.";
}


// ============================================================================
// TodayHourServerMap
//
// Creates and returns the string containing Date, hour, server name and map
// name
// ============================================================================

function string TodayHourServerMap(){
  local string Today;
  local string Hour;
  local string Server;
  local string Map;

  Today  = Level.Day@baseLogger.MonthStr(Level.Month)@Level.Year$", "$baseLogger.DayOfWeekStr(Level.DayOfWeek);
  Hour   = Level.Hour$":"$Level.Minute;
  Server = Level.Game.GameReplicationInfo.ServerName;
  Map    = Level.Title;

  return Today$"."@Hour$"."@Server$"."@Map$".";
}

function string GetThreeStars(GameReplicationInfo GRI){
  local string result;
  local int i, j;
  local array<PlayerReplicationInfo> PRIs;
  local PlayerReplicationInfo PRI,t;

  for (i=0;i<GRI.PRIArray.Length;i++)
  {
    PRI = GRI.PRIArray[i];

    // quick cascade sort to find out which player helped more to their team
    PRIs.Length = PRIs.Length+1;
    for (j=0;j<Pris.Length-1;j++)
    {
      // Warning!
      // Reading the following "if" statement may completely blow off your mind;
      // Summary:
      // if select the player who scored more by completeing objectives (score - kills = score_for_objectives)
      // if two or more players have the same score, check their frags (kills - suicides = frags)
      // if these two values are also equal, then check who died less
      if ( !PRIs[j].bOnlySpectator && PRIs[j].PlayerName != "WebAdmin" && PRIs[j].PlayerName != "DemoRecSpectator" && PRIS[j].Team != None && ((PRIs[j].Score - (PRIs[j].Kills - TeamPlayerReplicationInfo(PRIs[j]).Suicides) < (PRI.Score - (PRI.Kills - TeamPlayerReplicationInfo(PRI).Suicides) )) || ((PRIs[j].Score - (PRIs[j].Kills - TeamPlayerReplicationInfo(PRIs[j]).Suicides)) == (PRI.Score - (PRI.Kills - TeamPlayerReplicationInfo(PRI).Suicides)) && ((PRIs[j].Kills - TeamPlayerReplicationInfo(PRIs[j]).Suicides) < (PRI.Kills - TeamPlayerReplicationInfo(PRI).Suicides))) || ((PRIs[j].Score - (PRIs[j].Kills - TeamPlayerReplicationInfo(PRIs[j]).Suicides)) == (PRI.Score - (PRI.Kills - TeamPlayerReplicationInfo(PRI).Suicides)) && ( (PRIs[j].Kills - TeamPlayerReplicationInfo(PRIs[j]).Suicides) == (PRI.Kills - TeamPlayerReplicationInfo(PRI).Suicides)) ) &&  PRIs[j].Deaths > PRI.Deaths))
      {
        t = PRIs[j];
        PRIs[j] = PRI;
        PRI = t;
      }
    }
    PRIs[j] = PRI;
  }

  for (i = 0; i < PRIs.Length; i++){
    if ( Controller(PRIs[i].Owner).IsA('MessagingSpectator') || Controller(PRIs[i].Owner).IsA('DemoRecSpectator') || Controller(PRIs[i].Owner).IsInState('Spectating') || PRIs[i].Team == None ){
      if (bDebug){
        Log("Found a"$PRIs[i].playerName$", thinking he's a "$i+1$" star of the match. Removing that f*cker!"); //sorry for the language, but I got tired of seeing WebAdmin as the First star
      }
      PRIs.Remove(i, 1);
      i = i - 1; // so the loop won't skip the next entry.
      }
  }


  result = PRIs[0].PlayerName@"("$PRIs[0].Team.TeamName$"),";
  result = result@PRIs[1].PlayerName@"("$PRIs[1].Team.TeamName$"),";
  result = result@PRIs[2].PlayerName@"("$PRIs[2].Team.TeamName$").";

  return result;
}


// ============================================================================
// RemoveIllegalWindowsCharacters
//
// Takes the given string and removes the character that are not allowed for
// file names in Windows file systems. Called from GetLogFileName
// ============================================================================

function string StripIllegalWindowsCharacters (string result)
{
	ReplaceText(result,".", "-"); 			// Dots
	ReplaceText(result,"*", "-");				// Asterisks
	ReplaceText(result,":", "-");       // Colons
	ReplaceText(result,"|", "-");				// Vertical lines
	ReplaceText(result,"/", "-");				// Slashes
	ReplaceText(result,";", "-");				// Semicolons
	ReplaceText(result,">", "-");				// Less-than signs
	ReplaceText(result,"<", "-");				// Greater-than signs
	ReplaceText(result,"+", "-");				// Pluses
	ReplaceText(result,"?", "-");				// Question Marks
	ReplaceText(result,"\\","-");       // Backslashes (one more is added as the comiler interprest the
																			// only symbols as a mark that the next quotation mark is a part
																			// of the given string, not it's end;
	ReplaceText(result,"\"","_");				// Same for the quotation marks
	return result;
}


// ============================================================================
// GetLogFileName
//
// Takes the log file name mask, replaces vars in it and returns the
// ready-to-use string.
// ============================================================================

function string GetLogFilename()
{
  local string result;

  result = SummaryFileName;

  // replacing variables in file name mask
  ReplaceText(result, "%G", Level.Game.GameName);
  ReplaceText(result, "%N", Level.Game.GameReplicationInfo.ServerName);
  ReplaceText(result, "%A", Level.Title);
  ReplaceText(result, "%Y", Right("0000"$string(Level.Year), 4));
  ReplaceText(result, "%M", Right("00"$string(Level.Month), 2));
  ReplaceText(result, "%D", Right("00"$string(Level.Day), 2));
  ReplaceText(result, "%H", Right("00"$string(Level.Hour), 2));
  ReplaceText(result, "%I", Right("00"$string(Level.Minute), 2));
  ReplaceText(result, "%W", Right("0"$string(Level.DayOfWeek), 1));
  ReplaceText(result, "%S", Right("00"$string(Level.Second), 2));

  result = StripIllegalWindowsCharacters(result);

  return result;
}

// ============================================================================
// Shutdown
//
// Closes the file and broadcasts a message that the summary is saved
// ============================================================================
function Shutdown()
{
  if (TempLog!=None){
    TempLog.Destroy();
    Level.Game.Broadcast(self, "Game Summary "$GetLogFilename()$" Saved",);
  }
}

// Writing a string to a log
function Logf(string LogString)
{
  if (TempLog!=None)
    TempLog.Logf(LogString);
}


event Destroyed()
{
	Super.Destroyed();
}

DefaultProperties
{
  SummaryFileName="Summary_of_%G_at_%A_(%Y_%M_%D_%H_%I_%S)"
  HeadingsArray(0)=(Heading="%w shuts-out %l.",type=0)
  HeadingsArray(1)=(Heading="%l suffers humiliating defeat from %w.",type=0)
  HeadingsArray(2)=(Heading="%w: no regret for %l.",type=0)
  HeadingsArray(3)=(Heading="%l can't score against %w.",type=0)
  HeadingsArray(4)=(Heading="%w wins match against %l with a solid lead.",type=1)
  HeadingsArray(5)=(Heading="%w masterfully defeats %l.",type=1)
  HeadingsArray(6)=(Heading="%w defeats %l in a hard-fought battle.",type=2)
  HeadingsArray(7)=(Heading="%w wins over %l by one %p.",type=2)
  HeadingsArray(8)=(Heading="%w wins over %l by one %p in the overtime.",type=4)
  HeadingsArray(9)=(Heading="Capture in the overtime brings %w the victory over %l.",type=4)
  HeadingsArray(10)=(Heading="%w defeats %l.",type=6)
  HeadingsArray(11)=(Heading="%w is victorious over %l.",type=6)
  HeadingsArray(12)=(Heading="%w wins match against %l.",type=6)
  HeadingsArray(13)=(Heading="%l losts match against %w.",type=6)
  HeadingsArray(14)=(Heading="One %p brings %w the victory over %l.",type=2)
  HeadingsArray(15)=(Heading="%c's %p brings %w the victory over %l.",type=2)
  HeadingsArray(16)=(Heading="%c brings %w the victory over %l.",type=2)
  HeadingsArray(17)=(Heading="%w vs %l: both teams unable to score.",type=8)
  HeadingsArray(18)=(Heading="%w and %l couldn't find a winner.",type=7)
  HeadingsArray(19)=(Heading="No %ps, no assists as %w tied the match against %l.",type=8)
  HeadingsArray(20)=(Heading="%c's %p brings %w a draw in match against %l.",type=7)
  HeadingsArray(21)=(Heading="A rare draw match.",type=7)
  HeadingsArray(22)=(Heading="%c's %p brings %w the victory over %l in OT.",type=4)
  HeadingsArray(23)=(Heading="Both teams make %o %ps total as %w win over %l.",type=9)
  HeadingsArray(24)=(Heading="An all-out-attack game brings %w the victory over %l.",type=9)
  HeadingsArray(25)=(Heading="%c's %p brings %w the victory over %l in OT.",type=9)
  FriendlyName="GameLogger CTF Summary Creator"

  SummaryNameString="Summary File Name Mask"
  SectionName="CTF Summary Plugin"

  Description="Creates a very small summary of CTF match."
  Author="Michael \"_Lynx\" Sokolkov"
  version="0.8b"
  bDebug=True
}
