///////////////////////////////////////////////////////////////////////////////
// xLinkMeBot
// by Kamek (neokamek@gmail.com)
// Portions by Crusha K. Rool
// Extension of xBot with routines for link gun use.
// Bots will try to link up with players using the link gun.
// They will also attempt to heal critical vehicles (Mino, etc.)
///////////////////////////////////////////////////////////////////////////////

class xLinkMeBot extends xBot;

///////////////////////////////////////////////////////////////////////////////
// Vars, structs, consts, etc.
///////////////////////////////////////////////////////////////////////////////
const DAMAGED_VEHICLE_PCT = 0.55;			// If we see a big vehicle damaged below this amount of health, heal it
const DAMAGED_VEHICLE_HEAL = 0.75;			// How much we heal the damaged vehicle to if possible
const VEHICLE_PASSENGER_WAIT = 5.0;			// How long we'll wait for nearby passengers before driving off
const VEHICLE_PASSENGER_RADIUS = 10000;		// Radius for looking for vehicle passengers
const KILL_MINO_PCT = 0.25;					// Chance that we'll drop what we're doing and go kill the Mino
const KILL_MINO_LIFETIME = 120.0;			// If the Mino isn't destroyed by two minutes then "forget" where it is
											// This way bots don't turn into psychics and hunt it down no matter how
											// well you hide
											// This also allows us to call out the Mino again if it's on the move
											// and keep humans updated on its position.
const HEAL_NODE_RADIUS = 5000;				// Radius for looking for damaged nodes
const HEAL_NODE_PCT = 0.60;					// Chance that we'll heal a node
const LINK_HUMAN_PCT = 0.40;				// Chance that we'll link someone when noticed
const LINK_VEHICLE_PCT = 0.20;				// Chance that we'll link a damaged vehicle
const DEBUG_LOG = false;					// Log debug messages

var localized string MinoSpottedMessage;	// Message broadcast for when we've spotted a big tank

var Actor LinkTarget;						// Target that we're currently linking to
var Pawn EnemyMino;							// Contains a Pawn reference to the enemy's "big tank"
											// (usually a Mino, but could be something else like a Levi or Kraken)
var int EnemyMinoMaxHealth;					// Max Health of enemy's "big tank"
											// (used to determine whether a vehicle we see is actually the "big tank" of the level)
var bool bCheckingLinkTarget;				// If true, we're currently in CheckLinkTarget (avoid infinite recursion)
var bool bNowLinking;						// If true, we are currently linking to another player
var bool bAlreadyEnteredVehicle;			// If true, we're in a vehicle and already checked for passengers
var bool bMinoAttackSquad;					// Whether or not we want to kill this particular mino/big tank
var float VehicleWaitStart;					// Level.TimeSeconds we entered vehicle wait state
var float MinoSeenTime;						// Level.TimeSeconds we first spotted mino
var float RadarRange;						// Range of ONS hud radar

function DebugLog(string logme)
{
	if (DEBUG_LOG)
		log("["$Level.TimeSeconds$"]"@PlayerReplicationInfo.PlayerName@logme,'LinkMe Debug');
}

///////////////////////////////////////////////////////////////////////////////
// PostBeginPlay
// Determine the "big tank" of the level
// This is done by going through all vehicle factories and getting the max
// health of the vehicle spawned there. Then if we see a vehicle with max health
// equal or greater(?!) than this number, we'll call it out as the enemy mino.
///////////////////////////////////////////////////////////////////////////////
function PostBeginPlay()
{
	local SVehicleFactory Fact;
	local TerrainInfo T, PrimaryTerrain;

	Super.PostBeginPlay();

	foreach DynamicActors(class'SVehicleFactory', Fact)
	{
		if (Fact.VehicleClass.default.HealthMax > EnemyMinoMaxHealth)
			EnemyMinoMaxHealth = Fact.VehicleClass.default.HealthMax;
	}
	
	// Crusha K. Rool
	if (ONSOnslaughtGame(Level.Game) != None)
	{
		// Determine primary terrain
		foreach AllActors(class'TerrainInfo', T)
	    {
	        PrimaryTerrain = T;
	        if (T.Tag == 'PrimaryTerrain')
	            Break;
	    }
	
	    // Set RadarMaxRange to size of primary terrain
	    if (Level.bUseTerrainForRadarRange && PrimaryTerrain != None)
	        RadarRange = abs(PrimaryTerrain.TerrainScale.X * PrimaryTerrain.TerrainMap.USize) / 2.0;
	    else if (Level.CustomRadarRange > 0)
	        RadarRange = Clamp(Level.CustomRadarRange, 500.0, class'ONSHudOnslaught'.default.RadarMaxRange);
	}			
}

///////////////////////////////////////////////////////////////////////////////
// EnemyMinoSpotted
// Spotted enemy mino! Tell the team about it
///////////////////////////////////////////////////////////////////////////////
function EnemyMinoSpotted(SVehicle NewMino)
{
	local xLinkMeBot OtherBots;
	local string WarningMessage, LocationName, CardinalDirection, ItemName;
	local ONSPowerCore Node, NearestNode;
	local float Dist, MinDist;
	local xPickupBase Pick, NearestPick;

	// If we're the first to spot the mino, tell the whole team (bots and players)
	if (SetEnemyMino(NewMino))
	{
		foreach DynamicActors(class'xLinkMeBot', OtherBots)
		{
			// Only tell bots that are on our own team
			if (OtherBots.PlayerReplicationInfo.Team == PlayerReplicationInfo.Team)
				OtherBots.SetEnemyMino(NewMino);
		}

		// Find out exactly where we are
		if( ( NewMino.PlayerReplicationInfo.PlayerVolume != None ) && ( NewMino.PlayerReplicationInfo.PlayerVolume.LocationName != class'Volume'.Default.LocationName ) )
			LocationName = NewMino.PlayerReplicationInfo.PlayerVolume.LocationName;
		else if( NewMino.PlayerReplicationInfo.PlayerZone != None && ( NewMino.PlayerReplicationInfo.PlayerZone.LocationName != "" )  )
			LocationName = NewMino.PlayerReplicationInfo.PlayerZone.LocationName;

		// What's the closest power node
		if (ONSOnslaughtGame(Level.Game) != None)
		{
			MinDist = 10000;
			foreach DynamicActors(class'ONSPowerCore', Node)
			{
				Dist = VSize(Node.Location - NewMino.Location);
				if (Dist < MinDist)
				{
					MinDist = Dist;
					NearestNode = Node;
				}
			}
			if (NearestNode != None)
			{
				LocationName = LocationName @ "near "@NearestNode.GetHumanReadableName();
			}
			// If we're not near a powernode, determine mino location by cardinal direction
			// Crusha K. Rool
			else if (RadarRange != 0)
			{
				if (NewMino.Location.Y >= (RadarRange/3) && NewMino.Location.X < (RadarRange/-3))
					CardinalDirection = "Northwest";
				if (NewMino.Location.Y >= (RadarRange/3) && (NewMino.Location.X >= (RadarRange/-3) && NewMino.Location.X < (RadarRange/3)))
					CardinalDirection = "North";
				if (NewMino.Location.Y >= (RadarRange/3) && NewMino.Location.X >= (RadarRange/3))
					CardinalDirection = "Northeast";
				if ((NewMino.Location.Y < (RadarRange/3) && NewMino.Location.Y >= (RadarRange/-3)) && NewMino.Location.X < (RadarRange/-3))
					CardinalDirection = "West";
				if ((NewMino.Location.Y < (RadarRange/3) && NewMino.Location.Y >= (RadarRange/-3)) && (NewMino.Location.X >= (RadarRange/-3) && NewMino.Location.X < (RadarRange/3)))
					CardinalDirection = "Center";
				if ((NewMino.Location.Y < (RadarRange/3) && NewMino.Location.Y >= (RadarRange/-3)) && NewMino.Location.X >= (RadarRange/3))
					CardinalDirection = "East";
				if (NewMino.Location.Y < (RadarRange/-3) && NewMino.Location.X < (RadarRange/-3))
					CardinalDirection = "Southwest";
				if (NewMino.Location.Y < (RadarRange/-3) && (NewMino.Location.X >= (RadarRange/-3) && NewMino.Location.X < (RadarRange/3)))
					CardinalDirection = "South";
				if (NewMino.Location.Y < (RadarRange/-3) && NewMino.Location.X < (RadarRange/3))
					CardinalDirection = "Southeast";

				LocationName = LocationName @ CardinalDirection;
			}
		}

		// If mino's close to a big pickup, report the pickup
		MinDist = 5000;
		foreach DynamicActors(class'xPickupBase', Pick)
		{
			// Only consider: UDamage, Painter, Redeemer
			if (ClassIsChildOf(Pick.PowerUp, class'UDamagePack')
				|| (XWeaponBase(Pick) != None &&
					(ClassIsChildOf(XWeaponBase(Pick).WeaponType, class'Painter')
						|| ClassIsChildOf(XWeaponBase(Pick).WeaponType, class'Redeemer'))))
			{
				Dist = VSize(Pick.Location - NewMino.Location);
				if (Dist < MinDist)
				{
					MinDist = Dist;
					NearestPick = Pick;
				}
			}
		}
		if (NearestPick != None)
		{
			if (ClassIsChildOf(NearestPick.PowerUp, class'UDamagePack'))
				ItemName = "UDamage";
			else if (XWeaponBase(NearestPick) != None && ClassIsChildOf(XWeaponBase(NearestPick).WeaponType, class'Painter'))
				ItemName = "Target Painter";
			else if (XWeaponBase(NearestPick) != None && ClassIsChildOf(XWeaponBase(NearestPick).WeaponType, class'Redeemer'))
				ItemName = "Redeemer";

			if (ItemName != "")
				LocationName = LocationName @ "(near"@ItemName$")";
		}

		// Tell human players
		SendMessage(None, 'Other', GetMessageIndex('NEEDBACKUP'), 15, 'TEAM');

		WarningMessage = MinoSpottedMessage @ LocationName;
		ReplaceText(WarningMessage,"%V",NewMino.VehicleNameString);
	    Level.Game.BroadcastTeam( self, Level.Game.ParseMessageString( Level.Game.BaseMutator , self, WarningMessage ) , 'TeamSay');
	}
}

///////////////////////////////////////////////////////////////////////////////
// SetEnemyMino
// Enemy Mino has been spotted, go kill it
///////////////////////////////////////////////////////////////////////////////
function bool SetEnemyMino(SVehicle NewMino)
{
	if (EnemyMino == None)
	{
		EnemyMino = NewMino;
		MinoSeenTime = Level.TimeSeconds;

		// Don't send the entire team to kill the mino, just a few people.
		// If we're driving the mino then definitely go kill
		if (FRand() <= KILL_MINO_PCT || Pawn.IsA('Omnitaur'))
		{
			bMinoAttackSquad = true;
			KillEnemyMino();
		}
		else
			bMinoAttackSquad = false;

		return true;
	}
	return false;
}

///////////////////////////////////////////////////////////////////////////////
// CheckEnemyMino
// If mino destroyed, unset it so we don't keep attacking it
// Also unset it if enough time has passed
///////////////////////////////////////////////////////////////////////////////
function CheckEnemyMino()
{
	if (EnemyMino == None || EnemyMino.bDeleteMe || EnemyMino.Health <= 0 || Level.TimeSeconds - MinoSeenTime > KILL_MINO_LIFETIME)
	{
		EnemyMino = None;
		bMinoAttackSquad = false;
	}
}

///////////////////////////////////////////////////////////////////////////////
// KillEnemyMino
// Go try to kill the enemy mino
///////////////////////////////////////////////////////////////////////////////
function KillEnemyMino()
{
	local bool bDoKill;

	// If mino destroyed, unset it so we don't keep attacking it
	// Also unset it if enough time has passed
	CheckEnemyMino();

	if (EnemyMino != None)
	{
		// If we're in a fast vehicle (pretty much anything that's not a tank), go kill the mino now
		if (Vehicle(Pawn) != None && ONSHoverTank(Pawn) == None)
			bDoKill = true;
		// If we're in our own Mino and it's healthy, go kill the other team's Mino
		else if (Pawn.IsA('Omnitaur') && Pawn.Health / Pawn.HealthMax >= 0.75)
			bDoKill = true;
		// If we're NOT in a fast vehicle but we have visual contact with the mino, go kill it
		else
			bDoKill = FastTrace(EnemyMino.Location, Pawn.Location);

		if (bDoKill)
		{
			SetRouteToGoal(EnemyMino);
			Squad.SetEnemy(self, EnemyMino);
		}
	}
}

///////////////////////////////////////////////////////////////////////////////
// YellAt
// Tell idiot to stop shooting me
// Cut down on "same team" spam
///////////////////////////////////////////////////////////////////////////////
function YellAt(Pawn Moron)
{
	local float Threshold;

	if ( PlayerController(Moron.Controller) == None )
		Threshold = 1;
	else if ( Enemy == None )
		Threshold = 0.8;
	else
		Threshold = 0.9;
	if ( FRand() < Threshold )
		return;

	SendMessage(Moron.PlayerReplicationInfo, 'FRIENDLYFIRE', 0, 5, 'TEAM');
}

///////////////////////////////////////////////////////////////////////////////
// SetRouteToGoal
// If we're trying to link to something, ignore other routes
///////////////////////////////////////////////////////////////////////////////
function bool SetRouteToGoal(Actor A)
{
	if (LinkTarget != None && A != LinkTarget)
		return false;
	else
		return Super.SetRouteToGoal(A);
}

///////////////////////////////////////////////////////////////////////////////
// WhatToDoNext
// If we're linking, check to make sure we want to keep doing so.
///////////////////////////////////////////////////////////////////////////////
function ExecuteWhatToDoNext()
{
	local bool bExecuteSuper;
	local DestroyableObjective Node;

	bExecuteSuper = true;

	// See if we want to keep linking.
	CheckLinkTarget();

	// Ignore WhatToDoNext if we're linking. Finish the link first.
	if (LinkTarget == None)
	{
		// If there are nearby powernodes needing heals, do it
		// Check this BEFORE we hop in a vehicle and go kill the mino
		foreach VisibleCollidingActors(class'DestroyableObjective', Node, HEAL_NODE_RADIUS)
		{
			if (Node.TeamLink(GetTeamNum()) && Node.Health < Node.DamageCapacity && LinkTarget == None)
			{
				if (ONSPowerNode(Node) != None)
					DebugLog("Considering healing weak node"@Node.GetHumanReadableName());
				MaybeLink(Node);
			}
		}

		if (LinkTarget != None)
		{
			DebugLog("Trying to heal target"@LinkTarget);
			bExecuteSuper = false;
		}
		// Go kill the mino, if it exists.
		else if (EnemyMino != None)
		{
			if (bMinoAttackSquad)
			{
				bExecuteSuper = false;
				KillEnemyMino();
			}
			else
				CheckEnemyMino();
		}
		else
		{
			// If we just entered a vehicle, maybe wait for passengers
			if (!bAlreadyEnteredVehicle && Vehicle(Pawn) != None)
				bExecuteSuper = !MaybeWaitForPassengers();
			if (Vehicle(Pawn) == None)
				bAlreadyEnteredVehicle = false;
		}
	}
	else
		bExecuteSuper = false;
		
	// Final sanity check
	if (LinkTarget == None || LinkTarget.bDeleteMe)
		bExecuteSuper = true;

	if (bExecuteSuper)
		Super.ExecuteWhatToDoNext();
	else
		DebugLog("Not executing Super.WhatDoToNext -- Link target found"@LinkTarget);
}

///////////////////////////////////////////////////////////////////////////////
// MaybeWaitForPassengers
// Look around for potential passengers, and give 'em a honk
// Returns true if we're going to wait.
///////////////////////////////////////////////////////////////////////////////
function bool MaybeWaitForPassengers()
{
	local xPawn P;
	local int Passengers;
	local float Dist;

	DebugLog("Maybe wait for passengers");
	bAlreadyEnteredVehicle = true;

	// Don't bother waiting if we can't carry anybody.
	if (ONSVehicle(Pawn) == None || ONSVehicle(Pawn).WeaponPawns.Length == 0)
	{
		DebugLog("Not bothering -- Not an ONSVehicle, or can't hold passengers");
		return false;
	}

	// Look for potential passengers
	foreach DynamicActors(class'xPawn', P)
	{
		Dist = VSize(Pawn.Location - P.Location);
//		DebugLog("Considering passenger"@P.Name@": Health"@P.Health@"Team"@P.PlayerReplicationInfo.Team.Name@"Our Team:"@PlayerReplicationInfo.Team.Name@"Dist"@Dist);
//		if (/*Vehicle(P) == None && */P != Vehicle(Pawn).Driver && P.DrivenVehicle == None && P.Health > 0 && P.PlayerReplicationInfo != None && P.PlayerReplicationInfo.Team == PlayerReplicationInfo.Team && Dist <= VEHICLE_PASSENGER_RADIUS)
		if (P.IsHumanControlled() && Dist <= VEHICLE_PASSENGER_RADIUS)
		{
//			DebugLog("Found a potential passenger"@P.Name);
			Passengers++;
			break;
		}
	}

	// Honk the horn and wait a bit for one to possibly board
	if (Passengers > 0)
	{
		GotoState('VehicleWaitPassenger');
		return true;
	}
	else
	{
		DebugLog("Not bothering -- no passengers to wait for");
		return false;
	}
}

///////////////////////////////////////////////////////////////////////////////
// DoneLinking
// Done linking, move on with life as a bot
///////////////////////////////////////////////////////////////////////////////
function DoneLinking(optional string Reason)
{
	DebugLog("unsetting LinkTarget --"@Reason);
	LinkTarget = None;
	bNowLinking = False;
	ClearTemporaryOrders();
	WhatToDoNext(1);
}

///////////////////////////////////////////////////////////////////////////////
// SetTemporaryOrders
// If linking, ignore the orders and finish linking
///////////////////////////////////////////////////////////////////////////////
function SetTemporaryOrders(name NewOrders, Controller OrderGiver)
{
	if (LinkTarget == None)
		Super.SetTemporaryOrders(NewOrders, OrderGiver);
}

///////////////////////////////////////////////////////////////////////////////
// CheckLinkTarget
// See if we want to keep linking to this guy.
///////////////////////////////////////////////////////////////////////////////
function CheckLinkTarget()
{
	local float Dist;

	if (bCheckingLinkTarget)
		return;
	else
		bCheckingLinkTarget = true;

	if (LinkTarget != None)
	{
		// If it's a foot soldier, only continue to link if they are too
		if (Vehicle(LinkTarget) == None && Pawn(LinkTarget) != None)
		{
			if (!Pawn(LinkTarget).Weapon.IsA('LinkGun') || !Pawn(LinkTarget).Weapon.IsFiring())
				DoneLinking("Target's weapon is no longer firing or is not a Link Gun");
		}
		// If it's a vehicle, keep healing it until we're past the healing threshold
		else if (Pawn(LinkTarget) != None && Pawn(LinkTarget).Health / Pawn(LinkTarget).HealthMax > DAMAGED_VEHICLE_HEAL)
			DoneLinking("Vehicle has been healed");
		// If it's a node, keep healing it until full
		else if (DestroyableObjective(LinkTarget) != None && DestroyableObjective(LinkTarget).Health >= DestroyableObjective(LinkTarget).DamageCapacity)
			DoneLinking("Node has been healed");
		// If it's a node, stop trying to heal it if it's destroyed
		else if (DestroyableObjective(LinkTarget) != None && !DestroyableObjective(LinkTarget).TeamLink(GetTeamNum()))
			DoneLinking("Node has fallen");

		// If we're out of link ammo then knock it off
		if (FindLoadedLinkGun() == None)
			DoneLinking("No ammo left on our own link gun");

		// If we got attacked, deal with it
		if (Enemy != None && EnemyVisible())
			DoneLinking("Going to kill our enemy");
	}

	/*
	if (LinkTarget == None && OldOrders != 'None')
	{
		ClearTemporaryOrders();
		WhatToDoNext(1);
	}

	else */ if (LinkTarget != None)
	{
		Dist = VSize(Pawn.Location - LinkTarget.Location);

		if (Dist <= class'LinkFire'.Default.TraceRange * 0.75)
		{
			bNowLinking = True;
			if (ONSPowerNode(LinkTarget) != None && ONSPowerNode(LinkTarget).EnergySphere != None)
				DoRangedAttackOn(ONSPowerNode(LinkTarget).EnergySphere);
			else
				DoRangedAttackOn(LinkTarget);
		}
		/*
		if (!bNowLinking)
		{
			if (Dist <= class'LinkFire'.Default.TraceRange * 0.75)
			{
				bNowLinking = True;
				DoRangedAttackOn(LinkTarget);
			}
			else
				SetRouteToGoal(LinkTarget);
		}
		else */ if (Dist > class'LinkFire'.Default.TraceRange * 0.5)
		{
			// Move in closer
//			Target = LinkTarget;
			SetRouteToGoal(LinkTarget);
		}
	}

	bCheckingLinkTarget = false;
}

///////////////////////////////////////////////////////////////////////////////
// SwitchToThisWeapon
// Specifies a particular weapon for the bot to switch to.
// WARNING: Does not perform any sanity checks, such as for ammo, etc.
///////////////////////////////////////////////////////////////////////////////
function SwitchToThisWeapon(Weapon SwitchTo)
{
	if ( Pawn == None || Pawn.Inventory == None )
		return;

    Pawn.PendingWeapon = SwitchTo;
    if ( Pawn.PendingWeapon == Pawn.Weapon )
	    Pawn.PendingWeapon = None;
    if ( Pawn.PendingWeapon == None )
		return;

	StopFiring();

	if ( Pawn.Weapon == None )
		Pawn.ChangedWeapon();
	else if ( Pawn.Weapon != Pawn.PendingWeapon )
    {
		Pawn.Weapon.PutDown();
    }
}

///////////////////////////////////////////////////////////////////////////////
// SwitchToBestWeapon
// Don't try to switch weapons if we're currently linking a player.
// Otherwise, the bot will try to pull out a rocket launcher, etc. and shoot
// rockets at our link
///////////////////////////////////////////////////////////////////////////////
exec function SwitchToBestWeapon()
{
	if (LinkTarget == None)
		Super.SwitchToBestWeapon();
}

///////////////////////////////////////////////////////////////////////////////
// WeaponFireAgain
// Like original, but forces alt-fire on the link gun if we're linking.
///////////////////////////////////////////////////////////////////////////////
function bool WeaponFireAgain(float RefireRate, bool bFinishedFire)
{
	LastFireAttempt = Level.TimeSeconds;
	if ( Target == None )
		Target = Enemy;
	if ( Target != None )
	{
		if ( !Pawn.IsFiring() )
		{
			if ( (Pawn.Weapon != None && Pawn.Weapon.bMeleeWeapon) || (!NeedToTurn(Target.Location) && Pawn.CanAttack(Target)) )
			{
				Focus = Target;
				bCanFire = true;
				bStoppedFiring = false;
//				DebugLog("weaponfireagain LinkTarget"@LinkTarget@"weapon"@pawn.weapon);
				if (Pawn.Weapon != None && Pawn.Weapon.IsA('LinkGun') && LinkTarget != None)
					bFireSuccess = Pawn.Weapon.StartFire(1);		// Oh dear, what a nasty hack.
				else if (Pawn.Weapon != None)
					bFireSuccess = Pawn.Weapon.BotFire(bFinishedFire);
				else
				{
					Pawn.ChooseFireAt(Target);
					bFireSuccess = true;
				}
				return bFireSuccess;
			}
			else
			{
				bCanFire = false;
			}
		}
		else if ( bCanFire && ShouldFireAgain(RefireRate) )
		{
			if ( (Target != None) && (Focus == Target) && !Target.bDeleteMe )
			{
				bStoppedFiring = false;
				if (Pawn.Weapon != None && Pawn.Weapon.IsA('LinkGun') && LinkTarget != None)
					bFireSuccess = Pawn.Weapon.StartFire(1);
				else if (Pawn.Weapon != None)
					bFireSuccess = Pawn.Weapon.BotFire(bFinishedFire);
				else
				{
					Pawn.ChooseFireAt(Target);
					bFireSuccess = true;
				}
				return bFireSuccess;
			}
		}
	}
	StopFiring();
	return false;
}

///////////////////////////////////////////////////////////////////////////////
// LinkWith
// We've got our link gun out, now actually link to the target.
// Link target may not always be the actual target, it might be an intermediate
///////////////////////////////////////////////////////////////////////////////
function LinkWith(Actor SeenPlayer, optional pawn HumanPawn)
{
	local float Dist;
	local Controller OrderGiver;

	DebugLog("linking with"@SeenPlayer);

/*
	if (LinkTarget == None)
	{
		// Tell the player we're on our way
		if (PlayerController(SeenPlayer.Controller) != None)
			SendMessage(SeenPlayer.PlayerReplicationInfo, 'OTHER', GetMessageIndex('GOTYOURBACK'), 5, 'TEAM');

		// If we're healing a vehicle, let its driver know
		if (Vehicle(SeenPlayer) != None && Vehicle(SeenPlayer).Driver != None && PlayerController(Vehicle(SeenPlayer).Driver.Controller) != None)
			// Medic!
			SendMessage(Vehicle(SeenPlayer).Driver.PlayerReplicationInfo, 'OTHER', 30, 5, 'TEAM');
	}
*/

	LinkTarget = SeenPlayer;
	GoalString = "Link"@SeenPlayer;

	if (GetOrders() != 'FOLLOW' && Vehicle(SeenPlayer) == None && Pawn(SeenPlayer) != None)
	{
		if (HumanPawn != None)
			OrderGiver = HumanPawn.Controller;
		else
			OrderGiver = Pawn(SeenPlayer).Controller;

		if (OldOrders == 'None')
		{
			OldOrders = GetOrders();
			OldOrderGiver = Squad.SquadLeader;
			if (OldOrderGiver == None)
				OldOrderGiver = OrderGiver;
		}

		Aggressiveness = BaseAggressiveness + 1;
		UnrealTeamInfo(PlayerReplicationInfo.Team).AI.SetOrders(self,'FOLLOW',OrderGiver);
		WhatToDoNext(1);
	}

	// If out of range, move in closer
	Dist = VSize(Pawn.Location - SeenPlayer.Location);
//	Target = SeenPlayer;
	if (Dist > class'LinkFire'.Default.TraceRange * 0.75)
	{
//		log(self@"out of range -- moving in",'LinkMe');
		debuglog("WARNING!! Out of range, trying to move in");
		// FIXME -- Not actually moving toward target!
		if (!SetRouteToGoal(SeenPlayer))
			debuglog("WARNING!!! CANNOT MOVE TO GOAL!");
	}
	else
	{
//		log(self@"within range -- alt-firing link gun",'LinkMe');
		bNowLinking = true;
		if (Vehicle(SeenPlayer) != None && !SetRouteToGoal(SeenPlayer))
			debuglog("WARNING!!! CANNOT MOVE TO GOAL!");
		if (ONSPowerNode(SeenPlayer) != None && ONSPowerNode(SeenPlayer).EnergySphere != None)
			DoRangedAttackOn(ONSPowerNode(SeenPlayer).EnergySphere);
		else
			DoRangedAttackOn(SeenPlayer);
	}
}

///////////////////////////////////////////////////////////////////////////////
// FindLoadedLinkGun
// Returns a Weapon that is a Link Gun with ammo.
// Returns None if no Link Gun is available.
///////////////////////////////////////////////////////////////////////////////
function Weapon FindLoadedLinkGun()
{
	local Weapon Weap;
	local Inventory Inv;

	Weap = None;
	Inv = Pawn.Inventory;
	while (Inv != None && Weap == None)
	{
		if (Weapon(Inv) != None && Inv.IsA('LinkGun') && Weapon(Inv).HasAmmo())
			Weap = Weapon(Inv);
		Inv = Inv.Inventory;
	}

	if (Weap != None)
		return Weap;
	else
		return None;
}

///////////////////////////////////////////////////////////////////////////////
// TryToLink
// Try to use a link gun to link to this pawn.
///////////////////////////////////////////////////////////////////////////////
function TryToLink(Actor SeenPlayer)
{
	local Weapon Weap;
	local float Dist, MinDist;
	local Pawn P;
	local Actor LinkedActor;

	DebugLog("trying to link with"@SeenPlayer);

	// Never link if we're driving an important vehicle
	if (Vehicle(Pawn) != None && Vehicle(Pawn).HealthMax >= 1000)
	{
		debuglog("Link FAILED!! Reason: Driving a big vehicle");
		return;
	}

	// Never link if we're busy with an enemy
	if (Enemy != None && EnemyVisible())
	{
		debuglog("Link FAILED!! Reason: Busy with an enemy");
		return;
	}

	Dist = VSize(Pawn.Location - SeenPlayer.Location);
	MinDist = Dist;
	// If we're out of range, maybe there's a link chain we can join?
	// If it's a vehicle, prefer linking to another player first
	if (Dist > class'LinkFire'.Default.TraceRange || Vehicle(SeenPlayer) != None || DestroyableObjective(SeenPlayer) != None)
	{
		foreach DynamicActors(class'Pawn', P)
			// There's another pawn or pawns nearby us linking to something.
			// Assume that it's linking our desired target and link with the closest one
			if (P.Weapon != None && P.Weapon.IsA('LinkGun') && P.Weapon.IsFiring())
			{
				Dist = VSize(Pawn.Location - SeenPlayer.Location);
				if (Dist < MinDist && Dist <= class'LinkFire'.Default.TraceRange * 1.25)
				{
					MinDist = Dist;
					LinkedActor = P;
					debuglog("Found an intermediate link:"@P.PlayerReplicationInfo.PlayerName);
//					log(self@"found intermediate link"@LinkedActor,'LinkMe');
				}
			}
	}

	if (LinkedActor == None)
		LinkedActor = SeenPlayer;

	// Can we link, maybe?
	if (Pawn.Weapon != None && Pawn.Weapon.IsA('LinkGun'))
	{
		if (Pawn(SeenPlayer) != None)
			LinkWith(LinkedActor, Pawn(SeenPlayer));
		else
			LinkWith(LinkedActor);
	}
	else
	{
		// Look for a link gun and switch to it, THEN try to link
		Weap = FindLoadedLinkGun();

		// Switch to the link gun; fire if successful
		if (Weap != None)
		{
			SwitchToThisWeapon(Weap);
			if (Pawn.PendingWeapon != None && Pawn.PendingWeapon.IsA('LinkGun'))
			{
				if (Pawn(SeenPlayer) != None)
					LinkWith(LinkedActor, Pawn(SeenPlayer));
				else
					LinkWith(LinkedActor);
			}
			else
				debuglog("Link FAILED!! Reason: No valid link gun to link with.");
		}
		else
			debuglog("Link FAILED!! Reason: No valid link gun to link with.");
	}
}

///////////////////////////////////////////////////////////////////////////////
// MaybeLink
// Maybe link up with this guy?
///////////////////////////////////////////////////////////////////////////////
function MaybeLink(Actor SeenPlayer)
{
	local float Dist;

	// Sanity checks
	if (LinkTarget != None) // If already linking somebody
		return;
	if (SeenPlayer == None || SeenPlayer.bDeleteMe) // If link target is destroyed
		return;
	if (Pawn(SeenPlayer) != None &&	(Pawn(SeenPlayer).Health <= 0 || Pawn(SeenPlayer).PlayerReplicationInfo == None)) // If the other player is dead
		return;
	if (Pawn == None || Pawn.bDeleteMe || Pawn.Health <= 0 || PlayerReplicationInfo == None) // If WE'RE dead
		return;
	if (Pawn(SeenPlayer) != None && Pawn(SeenPlayer).PlayerReplicationInfo.Team != PlayerReplicationInfo.Team) // If they're not on our team
		return;
	if (Pawn(SeenPlayer) != None && !Pawn(SeenPlayer).IsHumanControlled() && SVehicle(SeenPlayer) == None) // If not human-controlled or a vehicle
		return;

	// Randomness checks
	if (DestroyableObjective(SeenPlayer) != None && DestroyableObjective(SeenPlayer).TeamLink(GetTeamNum()) && FRand() < HEAL_NODE_PCT)
		return;
	else if (Vehicle(SeenPlayer) != None && FRand() < LINK_VEHICLE_PCT)
		return;
	else if (Pawn(SeenPlayer) != None && FRand() < LINK_HUMAN_PCT)
		return;

//	log(self@"considering linking with"@SeenPlayer,'LinkMe');

	// If it's an ally, maybe link up with them
	// Make sure it's a human player so we don't end up in an endless link circlejerk
	// make sure we're within altfire range
	Dist = VSize(Pawn.Location - SeenPlayer.Location);

//	if (SeenPlayer.IsHumanControlled() && SeenPlayer.Weapon != None && SeenPlayer.Weapon.IsA('LinkGun') && SeenPlayer.Weapon.IsFiring())
//		DebugLog("found human firing link gun, trying to link. Dist is"@dist);

	if (Dist <= class'LinkFire'.Default.TraceRange * 3)
	{
		if (SVehicle(SeenPlayer) == None && Pawn(SeenPlayer) != None)
		{
			if (Pawn(SeenPlayer).Weapon.IsA('LinkGun') && Pawn(SeenPlayer).Weapon.IsFiring())
				TryToLink(SeenPlayer);
//			else if (SeenPlayer == LinkTarget)
//				LinkTarget = none;
		}
		// Link a damaged vehicle
		// Link even if it's not a human driving (the human might have hopped out to heal it, touch a node, etc)
		else if (SVehicle(SeenPlayer) != None && Pawn(SeenPlayer).Health / Pawn(SeenPlayer).HealthMax <= DAMAGED_VEHICLE_PCT)
			TryToLink(SeenPlayer);
		// Link a damaged node, no matter how much health is missing
		else if (DestroyableObjective(SeenPlayer) != None && DestroyableObjective(SeenPlayer).TeamLink(GetTeamNum()) && DestroyableObjective(SeenPlayer).Health < DestroyableObjective(SeenPlayer).DamageCapacity)
			TryToLink(SeenPlayer);
	}
}

///////////////////////////////////////////////////////////////////////////////
// SeePlayer
// We've seen a player, what do we do with it
///////////////////////////////////////////////////////////////////////////////
event SeePlayer(Pawn SeenPlayer)
{
	// If we're already linking, let's keep this link until the lead human is done. Ignore enemies and get the job done first
	CheckLinkTarget();

	if (LinkTarget == none)
	{
		// Look for enemy Mino
		if (SVehicle(SeenPlayer) != None && SeenPlayer.HealthMax >= EnemyMinoMaxHealth && SeenPlayer.PlayerReplicationInfo.Team != PlayerReplicationInfo.Team)
			EnemyMinoSpotted(SVehicle(SeenPlayer));

		// Look for enemies
		Super.SeePlayer(SeenPlayer);

		// Maybe link this guy
		MaybeLink(SeenPlayer);
	}
}

///////////////////////////////////////////////////////////////////////////////
// HearNoise
// We hear a noise. Maybe it's a link gun we want to link up with
///////////////////////////////////////////////////////////////////////////////
function HearNoise(float Loudness, Actor NoiseMaker)
{
	// If we're already linking, let's keep this link until the lead human is done. Ignore enemies and get the job done first
	CheckLinkTarget();

	if (LinkTarget == none)
	{
		// Look for enemies
		Super.HearNoise(Loudness, NoiseMaker);

		// Maybe link this guy
		MaybeLink(NoiseMaker.Instigator);
	}
}

///////////////////////////////////////////////////////////////////////////////
// Tick
// bad bad bad idea
///////////////////////////////////////////////////////////////////////////////
event Tick(float DeltaTime)
{
	local Pawn P;

//	log(self@"ticking",'LinkMe');

	// Ignore checks when dead
	// Also ignore checks if we're not set to FOLLOW
	if (Pawn == None || Pawn.bDeleteMe || Pawn.Health <= 0 || GetOrders() != 'FOLLOW')
		return;

	// If we're already linking, let's keep this link until the lead human is done. Ignore enemies and get the job done first
	CheckLinkTarget();

	if (LinkTarget == none)
	{
		// Look for link targets
		foreach VisibleCollidingActors(class'Pawn', P, class'LinkFire'.Default.TraceRange * 2)
			MaybeLink(P);
	}
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
// State VehicleWaitPassenger
// If we hop in a vehicle with empty seats, don't just speed off right away...
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
state VehicleWaitPassenger extends NoGoal
{
	ignores WhatToDoNext;

	// Honk the horn and wait for passengers
	event BeginState()
	{
		DebugLog("VehicleWaitPassenger BeginState");

		// Stop what you're doing and wait
		Target = None;
		MoveTarget = None;
		bAlreadyEnteredVehicle = true;
		VehicleWaitStart = Level.TimeSeconds;

		// Toot your horn and wait for people to get in
		if (Vehicle(Pawn) != None)
			Vehicle(Pawn).ServerPlayHorn(Rand(Vehicle(Pawn).HornSounds.Length - 1));
		SetTimer(0.5, true);

		/*
		// Look for potential passengers
		// If we find a bot on foot, maybe order it to get inside
		foreach VisibleCollidingActors(class'Pawn', P, VEHICLE_PASSENGER_RADIUS)
			if (Vehicle(P) == None && P.Health > 0 && P.PlayerReplicationInfo.Team == PlayerReplicationInfo.Team && Bot(P.Controller) != None && FRand() <= 0.25)
			{
				DebugLog("Ordering"@P.Controller.Name@"to get in");
				Bot(P.Controller).SetTemporaryOrders('FOLLOW', Self);
			}
		*/
	}

	// When seats fill, start drivin'
	event Timer()
	{
		local int i;
		local int SpotsLeft;

		// Figure out how many spots we have left
		if (ONSVehicle(Pawn) != None)
		{
			SpotsLeft = ONSVehicle(Pawn).WeaponPawns.Length;
			for (i = 0; i < ONSVehicle(Pawn).WeaponPawns.length; i++)
				if (ONSVehicle(Pawn).WeaponPawns[i].Controller != None)
				SpotsLeft--;
		}
		
		DebugLog("Vehicle wait timer check. Spots left"@SpotsLeft@"WeaponPawns Length"@ONSVehicle(Pawn).WeaponPawns.Length);

		// If no spots left, GO GO GO
		if (SpotsLeft <= 0)
			VehicleWaitStart = 0;

		// After we've waited a few seconds, GO GO GO
		if (Level.TimeSeconds - VehicleWaitStart > VEHICLE_PASSENGER_WAIT)
		{
			DebugLog("Done waiting around, go go go");
			GotoState('NoGoal');
			WhatToDoNext(55);
		}
	}
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
defaultproperties
{
	MinoSpottedMessage="Enemy %V"
}